# Database Sharing Guide

This guide explains how to share your database with team members for the kupi-mk project.

## 🎯 Quick Start

### For the Person Sharing the Database:
1. **Export your current database:**
   ```bash
   cd kupi-mk
   node scripts/export-database.js
   ```

2. **Commit the export to Git:**
   ```bash
   git add database-exports/latest-export.sql
   git commit -m "Add latest database export"
   git push
   ```

### For Team Members Getting the Database:
1. **Pull the latest changes:**
   ```bash
   git pull
   ```

2. **Import the database:**
   ```bash
   cd kupi-mk
   node scripts/import-database.js
   ```

## 📋 Detailed Instructions

### Setting Up Your Database (First Time)

1. **Install PostgreSQL** (if not already installed)
2. **Create the database and user:**
   ```sql
   CREATE DATABASE kupi_mk;
   CREATE USER admin WITH PASSWORD 'password123';
   GRANT ALL PRIVILEGES ON DATABASE kupi_mk TO admin;
   ```

3. **Import the shared database:**
   ```bash
   node scripts/import-database.js
   ```

### Exporting Your Database

When you want to share your current database state:

```bash
# Export current database
node scripts/export-database.js

# The script creates two files:
# - database-exports/database-export-[timestamp].sql (timestamped backup)
# - database-exports/latest-export.sql (for easy sharing)
```

### Importing a Database

To get the latest shared database:

```bash
# Import the latest export
node scripts/import-database.js

# Or import a specific export file
node scripts/import-database.js database-export-2025-01-15T10-30-00-000Z.sql
```

## 🔄 Workflow for Team Collaboration

### Daily Workflow:
1. **Start of day:** `git pull` and `node scripts/import-database.js`
2. **Work on your features**
3. **End of day (if you added data):** `node scripts/export-database.js` and commit

### When Someone Adds Important Data:
1. **Data creator:** Export and commit the database
2. **Team members:** Pull and import the new database

## 📁 File Structure

```
kupi-mk/
├── scripts/
│   ├── export-database.js    # Export script
│   └── import-database.js    # Import script
├── database-exports/
│   ├── latest-export.sql     # Always the newest export
│   └── database-export-*.sql # Timestamped backups
└── database_setup.sql        # Original schema
```

## ⚠️ Important Notes

1. **Always backup before importing** - The import script will DELETE existing data
2. **Images are NOT included** - Only database records are shared
3. **Coordinate with your team** - Avoid conflicting exports
4. **Test data only** - Don't use this for production databases

## 🚨 Troubleshooting

### Connection Issues:
```bash
# Make sure PostgreSQL is running
sudo systemctl status postgresql

# Check if you can connect
psql -U admin -d kupi_mk -h localhost
```

### Permission Issues:
```bash
# Make sure scripts are executable
chmod +x scripts/*.js
```

### Missing Dependencies:
```bash
# Install required packages
npm install
```

## 🎉 Alternative: Using Docker (Advanced)

For even easier sharing, you can use Docker:

1. **Create a docker-compose.yml** with your database
2. **Share the entire Docker setup**
3. **Team members just run `docker-compose up`**

This ensures everyone has the exact same environment!

---

## 🐳 Docker Database Synchronization

### Copy Local Database to Docker
Your local database has been successfully copied to Docker! Use these commands:

```bash
# Copy from local PostgreSQL to Docker
./copy-database-to-docker.sh

# Copy from Docker to local PostgreSQL  
./copy-database-from-docker.sh
```

### Current Database Status
- **Local Database**: localhost:5432 (your original data)
- **Docker Database**: localhost:5433 (Docker container)
- **Data Synchronized**: ✅ 27 products, 9 users, 10 orders

### Verification Commands
```bash
# Check Docker database
docker exec -it kupi-mk-postgres psql -U admin -d kupi_mk -c '\dt'

# Check record counts
docker exec -it kupi-mk-postgres psql -U admin -d kupi_mk -c "
SELECT 'products' as table_name, COUNT(*) FROM products 
UNION ALL SELECT 'users', COUNT(*) FROM users;
"
```

Your databases are now perfectly synchronized! 🎉
