<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Order extends Model
{
    use HasUuids;

    protected $primaryKey = 'id';
    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'date',
        'status',
        'estimated_delivery_date',
        'buyer_id',
        'receiver_id',
        'transport_id',
        'payment_id',
    ];

    public static function getValidationRules($id = null): array
    {
        return [
            'date' => 'required|date|before_or_equal:today',
            'status' => 'required|string|in:pending,confirmed,processing,shipped,delivered,cancelled',
            'estimated_delivery_date' => 'required|date|after_or_equal:date',
            'buyer_id' => 'required|string|exists:clients,id',
            'receiver_id' => 'required|string|exists:clients,id',
            'transport_id' => 'nullable|string|exists:transports,id',
            'payment_id' => 'nullable|string|exists:payments,id',
        ];
    }

    public function buyer(): BelongsTo
    {
        return $this->belongsTo(Client::class, 'buyer_id');
    }

    public function receiver(): BelongsTo
    {
        return $this->belongsTo(Client::class, 'receiver_id');
    }


    public function payment(): BelongsTo
    {
        return $this->belongsTo(Payment::class, 'payment_id');
    }

    public function transport(): BelongsTo
    {
        return $this->belongsTo(Transport::class);
    }

    public function invoice(): HasOne
    {
        return $this->hasOne(Invoice::class);
    }

    //ova ne mi treba deka ide kroz batches ne products
//    public function products(): BelongsToMany
//    {
//        return $this->belongsToMany(Product::class, 'order_products')
//            ->withPivot('quantity', 'price_per_unit', 'total_price')
//            ->withTimestamps();
//    }

    public function batches(): BelongsToMany
    {
        return $this->belongsToMany(Batch::class, 'order_batches')
            ->withPivot('quantity', 'price_per_unit', 'total_price')
            ->withTimestamps();
    }

    public function packingList(): HasOne
    {
        return $this->hasOne(PackingList::class);
    }

    public function getTotalAmountAttribute(): float
    {
        return $this->batches->sum('pivot.total_price') ?? 0;
    }
}
