<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
class GenericModelController extends Controller
{
    //helpers

    protected function normalizeForeignKeys(array $data, $modelInstance): array
    {
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id') && isset($data[$field]) && is_array($data[$field])) {
                // if frontend sent object instead of ID, extract it
                $data[$field] = $data[$field]['id'] ?? null;
            }
        }

        return $data;
    }

    public function index(Request $request, string $model)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }
        $query = $modelClass::query();
        $modelInstance = new $modelClass();
        $fillable = $modelInstance->getFillable();

        $relationships = $this->getRelationships($modelInstance);
        if (!empty($relationships)) {
            $query->with($relationships);
        }

        if ($request->has('search') && $request->search) {
            $query->where(function ($q) use ($request, $fillable) {
                foreach ($fillable as $field) {
                    $q->orWhere($field, 'like', '%' . $request->search . '%');
                }
            });
        }

        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        $items = $query->paginate(
            perPage: $request->get('per_page', 10),
            page: $request->get('page', 1)
        )->withQueryString();

        $relatedOptions = $this->getRelatedOptions($modelInstance);

        return Inertia::render('GenericModelPage', [
            'model' => Str::singular(Str::studly($model)),
            'items' => $items,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $request->get('per_page', 10),
                'open_id' => $request->get('open_id')
            ],
            'fkMap' => $this->getForeignKeyMap($modelInstance),
            'relatedOptions' => $relatedOptions,
        ]);
    }
//
    public function store(Request $request, string $model)
    {
        \Log::info('STORE - raw request payload', $request->all());
        \Log::info('STORE - model param: ' . $model);

        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $modelInstance = new $modelClass();

        // -- TEMP: bypass validation for debugging --
        // $validated = $request->validate($modelClass::getValidationRules());
        $data = $request->all();

        // If you have normalizeForeignKeys helper, use it. Otherwise inline simple normalization:
        if (method_exists($this, 'normalizeForeignKeys')) {
            $data = $this->normalizeForeignKeys($data, $modelInstance);
        } else {
            // simple inline normalization for objects like { producer: {id: 'uuid', name: 'X'} } -> producer_id
            $fillable = $modelInstance->getFillable();
            foreach ($fillable as $field) {
                if (Str::endsWith($field, '_id') && isset($data[$field]) && is_array($data[$field])) {
                    $data[$field] = $data[$field]['id'] ?? null;
                }
            }
        }

        \Log::info('STORE - normalized payload', $data);

        try {
            $created = $modelClass::create($data);
            \Log::info('STORE - created record', ['id' => $created->id ?? null, 'model' => $model]);
            // Return JSON so you can see success in the Network tab without redirect issues
//            return response()->json([
//                'status' => 'ok',
//                'created_id' => $created->id ?? null,
//                'data' => $data,
//            ], 201);
            return redirect()->route('generic.index', ['model' => $model])
                ->with('success', Str::singular($model) . ' created successfully.');
        } catch (\Throwable $e) {
            \Log::error('STORE - DB error: ' . $e->getMessage(), ['exception' => $e]);
//            return response()->json([
//                'status' => 'error',
//                'message' => $e->getMessage(),
//                'data' => $data,
//            ], 500);
            return redirect()->back()
                ->withErrors(['error' => 'Failed to create ' . $model . ': ' . $e->getMessage()])
                ->withInput();
        }
    }

    public function update(Request $request, string $model, string $id)
    {
        \Log::info('UPDATE - raw request payload', $request->all());
        \Log::info('UPDATE - model param: ' . $model . ' id: ' . $id);

        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);

        // -- TEMP: bypass validation for debugging --
        // $validated = $request->validate($modelClass::getValidationRules($item->id));
        $data = $request->all();

        if (method_exists($this, 'normalizeForeignKeys')) {
            $data = $this->normalizeForeignKeys($data, $item);
        } else {
            $fillable = $item->getFillable();
            foreach ($fillable as $field) {
                if (Str::endsWith($field, '_id') && isset($data[$field]) && is_array($data[$field])) {
                    $data[$field] = $data[$field]['id'] ?? null;
                }
            }
        }

        \Log::info('UPDATE - normalized payload', $data);

        try {
            $item->update($data);
            \Log::info('UPDATE - updated record', ['id' => $item->id]);
//            return response()->json([
//                'status' => 'ok',
//                'updated_id' => $item->id,
//                'data' => $data,
//            ], 200);
            return redirect()->route('generic.index', ['model' => $model])
                ->with('success', Str::singular($model) . ' updated successfully.');
        } catch (\Throwable $e) {
            \Log::error('UPDATE - DB error: ' . $e->getMessage(), ['exception' => $e]);
//            return response()->json([
//                'status' => 'error',
//                'message' => $e->getMessage(),
//                'data' => $data,
//            ], 500);
            return redirect()->back()
                ->withErrors(['error' => 'Failed to update ' . $model . ': ' . $e->getMessage()])
                ->withInput();
        }
    }


    public function destroy(string $model, string $id)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);
        $item->delete();

        return redirect()->route('generic.index', ['model' => $model])
            ->with('success', Str::singular($model) . ' deleted successfully.');
    }

    /**
     * Helper to resolve the model class.
     */
    protected function getModelClass(string $model)
    {
        $modelClass = 'App\\Models\\' . Str::studly(Str::singular($model));
        if (class_exists($modelClass)) {
            return $modelClass;
        }
        return null;
    }

    /**
     * Get relationships for eager loading based on foreign keys
     */
    protected function getRelationships($modelInstance): array
    {
        $relationships = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');

                // Check if the relationship method exists
                if (method_exists($modelInstance, $relationName)) {
                    try {
                        $relation = $modelInstance->$relationName();
                        if ($relation instanceof BelongsTo) {
                            $relationships[] = $relationName;
                        }
                    } catch (\Exception $e) {
                        // Skip if the relationship doesn't exist or has issues
                        continue;
                    }
                }
            }
        }

        return $relationships;
    }

    /**
     * Generate foreign key mapping for the frontend
     */
    protected function getForeignKeyMap($modelInstance): array
    {
        $fkMap = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');

                // Map to plural table name (following Laravel conventions)
                $tableName = Str::plural($relationName);
                $fkMap[$field] = $tableName;
            }
        }

        return $fkMap;
    }

    protected function getRelatedOptions($modelInstance): array
    {
        $options = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');
                if (method_exists($modelInstance, $relationName)) {
                    try {
                        $relation = $modelInstance->$relationName();
                        if ($relation instanceof BelongsTo) {
                            $relatedModel = $relation->getRelated();
                            $relatedItems = $relatedModel->all();

                            $options[$field] = $relatedItems->map(function ($item) {
                                return [
                                    'id' => $item->id,
                                    'name' => $item->name ?? ("#{$item->id}"),
                                ];
                            })->values();
                        }
                    } catch (\Exception $e) {
                        continue;
                    }
                }
            }
        }

        return $options;
    }
}

//
//namespace App\Http\Controllers;
//
//use Illuminate\Http\Request;
//use Inertia\Inertia;
//use Illuminate\Support\Str;
//use Illuminate\Database\Eloquent\Relations\BelongsTo;
//use function Laravel\Prompts\error;
//
//class GenericModelController extends Controller
//{
//    protected function preprocessFormData(array $data, $modelInstance): array
//    {
//        $fillable = $modelInstance->getFillable();
//
//        foreach ($fillable as $field) {
//            // hendla nadvoreshni kluchevi
//            if (Str::endsWith($field, '_id') && isset($data[$field])) {
//                // ekstrahira id of objektot ako e daden kako objekt
//                if (is_array($data[$field])) {
//                    $data[$field] = $data[$field]['id'] ?? null;
//                }
//                // konvertira prazen string u null za nadvoreshni kluchevi
//                if ($data[$field] === '') {
//                    $data[$field] = null;
//                }
//            }
//
//            // hendla boolean polinja
//            if (isset($data[$field]) && is_bool($modelInstance->$field)) {
//                $data[$field] = (bool) $data[$field];
//
//            }
//
//            // hendla date polinja
//            $casts = $modelInstance->getCasts();
//            if (isset($casts[$field]) && strpos($casts[$field], 'date') !== false && isset($data[$field])) {
//                if ($data[$field] === '') {
//                    $data[$field] = null;
//                }
//            }
//
//            return $data;
//        }
//    }
//
//    public function store(Request $request, string $model)
//    {
//        \Log::info('STORE - raw request payload', $request->all());
//        \Log::info('STORE - model param: ' . $model);
//
//        $modelClass = $this->getModelClass($model);
//        if (!$modelClass) {
//            abort(404, 'Model not found.');
//        }
//
//        $modelInstance = new $modelClass();
//
//
//        $data = $request->all();
//        $data = $this->preprocessFormData($data, $modelInstance);
//
//        try {
//            $valdated = validator($data, $modelClass:getValidationRules()) -> validate();
//        } catch (\Illuminate\Validation\ValidationException $e) {
//            log.error('STORE - validation failed', $e->errors());
//        }
//
////        // Re-enable validation but with better error handling
////        try {
////            $validated = $request->validate($data, $modelClass::getValidationRules());
////            \Log::info('STORE - validation passed', $validated);
////        } catch (\Illuminate\Validation\ValidationException $e) {
////            \Log::error('STORE - validation failed', $e->errors());
////
////            // Return proper Inertia error response
////            return redirect()->back()
////                ->withErrors($e->errors())
////                ->withInput();
////        }
//
//        // Normalize foreign keys
//        $data = $this->normalizeForeignKeys($validated, $modelInstance);
//        \Log::info('STORE - normalized payload', $data);
//
//        try {
//            $created = $modelClass::create($data);
//            \Log::info('STORE - created record', ['id' => $created->id ?? null, 'model' => $model]);
//
//            // Proper Inertia redirect after successful creation
//            return redirect()->route('generic.index', ['model' => $model])
//                ->with('success', Str::singular($model) . ' created successfully.');
//
//        } catch (\Throwable $e) {
//            \Log::error('STORE - DB error: ' . $e->getMessage(), ['exception' => $e]);
//
//            return redirect()->back()
//                ->withErrors(['error' => 'Failed to create ' . $model . ': ' . $e->getMessage()])
//                ->withInput();
//        }
//    }
//
//    public function update(Request $request, string $model, string $id)
//    {
//        \Log::info('UPDATE - raw request payload', $request->all());
//        \Log::info('UPDATE - model param: ' . $model . ' id: ' . $id);
//
//        $modelClass = $this->getModelClass($model);
//        if (!$modelClass) {
//            abort(404, 'Model not found.');
//        }
//
//        $item = $modelClass::findOrFail($id);
//
//        // Re-enable validation
//        try {
//            $validated = $request->validate($modelClass::getValidationRules($item->id));
//            \Log::info('UPDATE - validation passed', $validated);
//        } catch (\Illuminate\Validation\ValidationException $e) {
//            \Log::error('UPDATE - validation failed', $e->errors());
//
//            return redirect()->back()
//                ->withErrors($e->errors())
//                ->withInput();
//        }
//
//        $data = $this->normalizeForeignKeys($validated, $item);
//        \Log::info('UPDATE - normalized payload', $data);
//
//        try {
//            $item->update($data);
//            \Log::info('UPDATE - updated record', ['id' => $item->id]);
//
//            return redirect()->route('generic.index', ['model' => $model])
//                ->with('success', Str::singular($model) . ' updated successfully.');
//
//        } catch (\Throwable $e) {
//            \Log::error('UPDATE - DB error: ' . $e->getMessage(), ['exception' => $e]);
//
//            return redirect()->back()
//                ->withErrors(['error' => 'Failed to update ' . $model . ': ' . $e->getMessage()])
//                ->withInput();
//        }
//    }
//
//    // Improved normalizeForeignKeys method
//    protected function normalizeForeignKeys(array $data, $modelInstance): array
//    {
//        $fillable = $modelInstance->getFillable();
//
//        foreach ($fillable as $field) {
//            if (Str::endsWith($field, '_id') && isset($data[$field])) {
//                // Handle various input formats
//                if (is_array($data[$field])) {
//                    // Extract ID from object: {id: 'uuid', name: 'Name'}
//                    $data[$field] = $data[$field]['id'] ?? null;
//                } elseif (is_string($data[$field]) && empty($data[$field])) {
//                    // Convert empty strings to null for foreign keys
//                    $data[$field] = null;
//                }
//                // If it's already a valid ID (string/int), leave it as is
//            }
//        }
//
//        return $data;
//    }
//
//    // Rest of your existing methods remain the same...
//
//    protected function getModelClass(string $model)
//    {
//        $modelClass = 'App\\Models\\' . Str::studly(Str::singular($model));
//        if (class_exists($modelClass)) {
//            return $modelClass;
//        }
//        return null;
//    }
//
//    protected function getRelationships($modelInstance): array
//    {
//        $relationships = [];
//        $fillable = $modelInstance->getFillable();
//
//        foreach ($fillable as $field) {
//            if (Str::endsWith($field, '_id')) {
//                $relationName = Str::beforeLast($field, '_id');
//
//                if (method_exists($modelInstance, $relationName)) {
//                    try {
//                        $relation = $modelInstance->$relationName();
//                        if ($relation instanceof BelongsTo) {
//                            $relationships[] = $relationName;
//                        }
//                    } catch (\Exception $e) {
//                        continue;
//                    }
//                }
//            }
//        }
//
//        return $relationships;
//    }
//
//    protected function getForeignKeyMap($modelInstance): array
//    {
//        $fkMap = [];
//        $fillable = $modelInstance->getFillable();
//
//        foreach ($fillable as $field) {
//            if (Str::endsWith($field, '_id')) {
//                $relationName = Str::beforeLast($field, '_id');
//                $tableName = Str::plural($relationName);
//                $fkMap[$field] = $tableName;
//            }
//        }
//
//        return $fkMap;
//    }
//
//    protected function getRelatedOptions($modelInstance): array
//    {
//        $options = [];
//        $fillable = $modelInstance->getFillable();
//
//        foreach ($fillable as $field) {
//            if (Str::endsWith($field, '_id')) {
//                $relationName = Str::beforeLast($field, '_id');
//                if (method_exists($modelInstance, $relationName)) {
//                    try {
//                        $relation = $modelInstance->$relationName();
//                        if ($relation instanceof BelongsTo) {
//                            $relatedModel = $relation->getRelated();
//                            $relatedItems = $relatedModel->all();
//
//                            $options[$field] = $relatedItems->map(function ($item) {
//                                return [
//                                    'id' => $item->id,
//                                    'name' => $item->name ?? ("#{$item->id}"),
//                                ];
//                            })->values();
//                        }
//                    } catch (\Exception $e) {
//                        continue;
//                    }
//                }
//            }
//        }
//
//        return $options;
//    }
//
//    public function index(Request $request, string $model)
//    {
//        $modelClass = $this->getModelClass($model);
//        if (!$modelClass) {
//            abort(404, 'Model not found.');
//        }
//
//        $query = $modelClass::query();
//        $modelInstance = new $modelClass();
//        $fillable = $modelInstance->getFillable();
//
//        $relationships = $this->getRelationships($modelInstance);
//        if (!empty($relationships)) {
//            $query->with($relationships);
//        }
//
//        if ($request->has('search') && $request->search) {
//            $query->where(function ($q) use ($request, $fillable) {
//                foreach ($fillable as $field) {
//                    $q->orWhere($field, 'like', '%' . $request->search . '%');
//                }
//            });
//        }
//
//        $sortBy = $request->get('sort_by', 'created_at');
//        $sortDirection = $request->get('sort_direction', 'desc');
//        $query->orderBy($sortBy, $sortDirection);
//
//        $items = $query->paginate(
//            perPage: $request->get('per_page', 10),
//            page: $request->get('page', 1)
//        )->withQueryString();
//
//        $relatedOptions = $this->getRelatedOptions($modelInstance);
//
//        return Inertia::render('GenericModelPage', [
//            'model' => Str::singular(Str::studly($model)),
//            'items' => $items,
//            'filters' => [
//                'search' => $request->search,
//                'sort_by' => $sortBy,
//                'sort_direction' => $sortDirection,
//                'per_page' => $request->get('per_page', 10),
//                'open_id' => $request->get('open_id')
//            ],
//            'fkMap' => $this->getForeignKeyMap($modelInstance),
//            'relatedOptions' => $relatedOptions,
//        ]);
//    }
//
//    public function destroy(string $model, string $id)
//    {
//        $modelClass = $this->getModelClass($model);
//        if (!$modelClass) {
//            abort(404, 'Model not found.');
//        }
//
//        $item = $modelClass::findOrFail($id);
//        $item->delete();
//
//        return redirect()->route('generic.index', ['model' => $model])
//            ->with('success', Str::singular($model) . ' deleted successfully.');
//    }
//}
