<?php

namespace Database\Seeders;

use App\Models\Order;
use App\Models\Client;
use App\Models\Transport;
use App\Models\Batch;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class OrderSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $buyers = Client::all();
        $receivers = Client::all();
        $transports = Transport::all();
        $batches = Batch::with('product')->get();

        if ($buyers->isEmpty() || $receivers->isEmpty() || $transports->isEmpty() || $batches->isEmpty()) {
            $this->command->warn('No clients, transports, or batches found. Please seed these first.');
            return;
        }

        $statuses = ['pending', 'confirmed', 'processing', 'shipped', 'delivered', 'cancelled'];

        for ($i = 0; $i < 20; $i++) {
            $orderDate = now()->subDays(rand(1, 60));
            $estimatedDeliveryDate = $orderDate->copy()->addDays(rand(7, 30));

            $order = Order::create([
                'id' => Str::uuid(),
                'date' => $orderDate,
                'status' => $statuses[array_rand($statuses)],
                'estimated_delivery_date' => $estimatedDeliveryDate,
                'buyer_id' => $buyers->random()->id,
                'receiver_id' => $receivers->random()->id,
                'transport_id' => $transports->random()->id,
            ]);

            // Attach 1-3 random batches to this order
            $selectedBatches = $batches->random(rand(1, min(3, $batches->count())));

            foreach ($selectedBatches as $batch) {
                $quantity = rand(100, 1000);
                $pricePerUnit = $batch->product->price ?? rand(10, 100);
                $totalPrice = $quantity * $pricePerUnit;

                DB::table('order_batches')->insert([
                    'id' => (string)Str::uuid(),
                    'order_id' => $order->id,
                    'batch_id' => $batch->id,
                    'quantity' => $quantity,
                    'price_per_unit' => $pricePerUnit,
                    'total_price' => $totalPrice,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }

        $this->command->info('Orders with batches seeded successfully!');
    }
}
