<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
class GenericModelController extends Controller
{
    public function index(Request $request, string $model)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }
        $query = $modelClass::query();
        $modelInstance = new $modelClass();
        $fillable = $modelInstance->getFillable();

        $relationships = $this->getRelationships($modelInstance);
        if (!empty($relationships)) {
            $query->with($relationships);
        }

        if ($request->has('search') && $request->search) {
            $query->where(function ($q) use ($request, $fillable) {
                foreach ($fillable as $field) {
                    $q->orWhere($field, 'like', '%' . $request->search . '%');
                }
            });
        }

        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        $items = $query->paginate(
            perPage: $request->get('per_page', 10),
            page: $request->get('page', 1)
        )->withQueryString();

        $relatedOptions = $this->getRelatedOptions($modelInstance);

        return Inertia::render('GenericModelPage', [
            'model' => Str::singular(Str::studly($model)),
            'items' => $items,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $request->get('per_page', 10),
                'open_id' => $request->get('open_id')
            ],
            'fkMap' => $this->getForeignKeyMap($modelInstance),
            'relatedOptions' => $relatedOptions,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, string $model)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $modelInstance = new $modelClass();
        $fillable = $modelInstance->getFillable();

        // Dynamically get validation rules from a method on the model
        $validated = $request->validate($modelClass::getValidationRules());

        $modelClass::create($validated);

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' created successfully.');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $model, string $id)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);

        $validated = $request->validate($modelClass::getValidationRules($item->id));

        $item->update($validated);

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $model, string $id)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);
        $item->delete();

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' deleted successfully.');
    }

    /**
     * Helper to resolve the model class.
     */
    protected function getModelClass(string $model)
    {
        $modelClass = 'App\\Models\\' . Str::studly(Str::singular($model));
        if (class_exists($modelClass)) {
            return $modelClass;
        }
        return null;
    }

    /**
     * Get relationships for eager loading based on foreign keys
     */
    protected function getRelationships($modelInstance): array
    {
        $relationships = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');

                // Check if the relationship method exists
                if (method_exists($modelInstance, $relationName)) {
                    try {
                        $relation = $modelInstance->$relationName();
                        if ($relation instanceof BelongsTo) {
                            $relationships[] = $relationName;
                        }
                    } catch (\Exception $e) {
                        // Skip if relationship doesn't exist or has issues
                        continue;
                    }
                }
            }
        }

        return $relationships;
    }

    /**
     * Generate foreign key mapping for the frontend
     */
    protected function getForeignKeyMap($modelInstance): array
    {
        $fkMap = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');

                // Map to plural table name (following Laravel conventions)
                $tableName = Str::plural($relationName);
                $fkMap[$field] = $tableName;
            }
        }

        return $fkMap;
    }

    protected function getRelatedOptions($modelInstance): array
    {
        $options = [];
        $fillable = $modelInstance->getFillable();

        foreach ($fillable as $field) {
            if (Str::endsWith($field, '_id')) {
                $relationName = Str::beforeLast($field, '_id');
                if (method_exists($modelInstance, $relationName)) {
                    try {
                        $relation = $modelInstance->$relationName();
                        if ($relation instanceof BelongsTo) {
                            $relatedModel = $relation->getRelated();
                            $relatedItems = $relatedModel->all();

                            $options[$field] = $relatedItems->map(function ($item) {
                                return [
                                    'id' => $item->id,
                                    'name' => $item->name ?? ("#{$item->id}"),
                                ];
                            })->values();
                        }
                    } catch (\Exception $e) {
                        continue;
                    }
                }
            }
        }

        return $options;
    }
}
