<?php

namespace App\Http\Controllers;

use App\Models\Client;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ClientController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Client::query();

        // Add search functionality
        if ($request->has('search') && $request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('client_name', 'like', '%' . $request->search . '%')
                    ->orWhere('country', 'like', '%' . $request->search . '%')
                    ->orWhere('registration_number', 'like', '%' . $request->search . '%')
                    ->orWhere('tax_id', 'like', '%' . $request->search . '%')
                    ->orWhere('contact_person', 'like', '%' . $request->search . '%')
                    ->orWhere('phone_number', 'like', '%' . $request->search . '%')
                    ->orWhere('billing_address', 'like', '%' . $request->search . '%')
                    ->orWhere('shipping_address', 'like', '%' . $request->search . '%');
            });
        }

        // Add sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        // Paginate results
        $clients = $query->paginate(
            perPage: $request->get('per_page', 10),
            page: $request->get('page', 1)
        )->withQueryString(); // Preserve query parameters

        return Inertia::render('Client', [
            'clients' => $clients,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $request->get('per_page', 10)
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //return Inertia::render('Client/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'client_name' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'registration_number' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'phone_number' => 'nullable|string|max:255',
            'billing_address' => 'nullable|string',
            'shipping_address' => 'nullable|string',
        ]);

        Client::create($validated);

        return redirect()->route('clients.index')
            ->with('success', 'Client created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
//        $client = Client::with(['buyerOrders', 'receiverOrders'])->findOrFail($id);
//
//        return Inertia::render('Client/Show', [
//            'client' => $client,
//        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
//        $client = Client::findOrFail($id);
//
//        return Inertia::render('Client/Edit', [
//            'client' => $client,
//        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $client = Client::findOrFail($id);

        $validated = $request->validate([
            'client_name' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'registration_number' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'phone_number' => 'nullable|string|max:255',
            'billing_address' => 'nullable|string',
            'shipping_address' => 'nullable|string',
        ]);

        $client->update($validated);

        return redirect()->route('clients.index')
            ->with('success', 'Client updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
//    public function destroy(string $id)
//    {
//        $client = Client::findOrFail($id);
//
//        // Check if the client has any orders before deletion
//        if ($client->buyerOrders()->exists() || $client->receiverOrders()->exists()) {
//            return redirect()->route('clients.index')
//                ->with('error', 'Cannot delete client with existing orders.');
//        }
//
//        $client->delete();
//
//        return redirect()->route('clients.index')
//            ->with('success', 'Client deleted successfully.');
//    }

    /**
     * Get clients for dropdowns/select options
     */
    public function getClientsForSelect(Request $request)
    {
        $query = Client::select('id', 'client_name', 'country');

        if ($request->has('search') && $request->search) {
            $query->where('client_name', 'like', '%' . $request->search . '%');
        }

        $clients = $query->orderBy('client_name')->limit(50)->get();

        return response()->json($clients);
    }
}
