<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Payment;
use App\Models\Client;
use App\Models\Transport;
use App\Models\Batch;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class OrderController extends Controller
{
    /**
     * Display a listing of orders - This is called from GenericModelController
     */
    public function index(Request $request)
    {
        $model = 'Order';

        $query = Order::with([
            'buyer',
            'receiver',
            'transport',
            'payment',
            'batches' => function($query) {
                $query->with('product');
            }
        ]);

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                    ->orWhere('status', 'like', "%{$search}%")
                    ->orWhereHas('buyer', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('receiver', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        $sortBy = $request->get('sort_by', 'date');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        $perPage = $request->get('per_page', 10);
        $items = $query->paginate($perPage);

        $fkMap = [
            'buyer_id' => 'clients',
            'receiver_id' => 'clients',
            'transport_id' => 'transports',
        ];

        $relatedOptions = [
            'buyer_id' => Client::select('id', 'name')->get()->toArray(),
            'receiver_id' => Client::select('id', 'name')->get()->toArray(),
            'transport_id' => Transport::select('id', 'name')->get()->toArray(),
        ];

        // Get available batches for the create form
        $availableBatches = Batch::with('product')->get()->map(function($batch) {
            return [
                'id' => $batch->id,
                'batch_code' => $batch->batch_code,
                'product_name' => $batch->product->name ?? 'N/A',
                'units_per_batch' => $batch->units_per_batch,
                'net_weight' => $batch->net_weight,
            ];
        });

        return Inertia::render('OrderPage', [
            'model' => $model,
            'items' => $items,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $perPage,
                'open_id' => $request->open_id,
            ],
            'fkMap' => $fkMap,
            'relatedOptions' => $relatedOptions,
            'availableBatches' => $availableBatches,
        ]);
    }

    /**
     * Store a newly created order
     */
    public function store(Request $request)
    {
        $validated = $request->validate(Order::getValidationRules());

        try {
            DB::beginTransaction();

            $order = Order::create($validated);

            // Add batches if provided
            if ($request->has('batches') && is_array($request->batches)) {
                foreach ($request->batches as $batchData) {
                    $order->batches()->attach($batchData['batch_id'], [
                        'quantity' => $batchData['quantity'],
                        'price_per_unit' => $batchData['price_per_unit'],
                        'total_price' => $batchData['quantity'] * $batchData['price_per_unit']
                    ]);
                }
            }

            // Create payment if requested
            if ($request->boolean('create_payment')) {
                $totalAmount = 0;
                if ($request->has('batches') && is_array($request->batches)) {
                    foreach ($request->batches as $batchData) {
                        $totalAmount += $batchData['quantity'] * $batchData['price_per_unit'];
                    }
                }

                Payment::create([
                    'order_id' => $order->id,
                    'amount' => $totalAmount,
                    'currency' => $request->get('payment_currency', 'USD'),
                    'due_date' => $request->get('payment_due_date', now()->addDays(30)),
                    'exchange_rate' => $request->get('payment_exchange_rate', 1.0),
                    'payment_date' => null,
                    'payment_method' => $request->get('payment_method', 'bank_transfer'),
                    'payment_status' => 'pending'
                ]);
            }

            DB::commit();

            return redirect()->route('generic.index', ['model' => 'orders'])
                ->with('success', 'Order created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create order: ' . $e->getMessage());
        }
    }

    /**
     * Update an existing order
     */
    public function update(Request $request, Order $order)
    {
        $validated = $request->validate(Order::getValidationRules($order->id));

        try {
            DB::beginTransaction();

            $order->update($validated);

            // Update batches if provided
            if ($request->has('batches')) {
                $order->batches()->detach();
                foreach ($request->batches as $batchData) {
                    $order->batches()->attach($batchData['batch_id'], [
                        'quantity' => $batchData['quantity'],
                        'price_per_unit' => $batchData['price_per_unit'],
                        'total_price' => $batchData['quantity'] * $batchData['price_per_unit']
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('generic.index', ['model' => 'orders'])
                ->with('success', 'Order updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to update order: ' . $e->getMessage());
        }
    }

    /**
     * Delete an order
     */
    public function destroy(Order $order)
    {
        try {
            DB::beginTransaction();

            $order->batches()->detach();
            $order->delete();

            DB::commit();

            return redirect()->route('generic.index', ['model' => 'orders'])
                ->with('success', 'Order deleted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete order: ' . $e->getMessage());
        }
    }

    /**
     * Generate and download invoice PDF for an order
     */
    public function generateInvoice(Order $order)
    {
        try {
            // Load all necessary relationships
            $order->load([
                'buyer',
                'receiver',
                'transport',
                'payment',
                'batches.product'
            ]);

            // Calculate totals
            $subtotal = $order->batches->sum(function($batch) {
                return $batch->pivot->quantity * $batch->pivot->price_per_unit;
            });

            $taxRate = 18; // 18% tax - adjust as needed
            $taxAmount = $subtotal * ($taxRate / 100);
            $totalAmount = $subtotal + $taxAmount;

            // Generate invoice number
            $invoiceNumber = 'INV-' . date('Y') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            // Prepare data for PDF
            $data = [
                'order' => $order,
                'invoiceNumber' => $invoiceNumber,
                'invoiceDate' => now()->format('Y-m-d'),
                'dueDate' => now()->addDays(30)->format('Y-m-d'),
                'subtotal' => number_format($subtotal, 2),
                'taxRate' => $taxRate,
                'taxAmount' => number_format($taxAmount, 2),
                'totalAmount' => number_format($totalAmount, 2),
                'currency' => $order->payment->currency ?? 'USD',
            ];

            // Generate PDF with proper headers for download
            $pdf = Pdf::loadView('pdfs.invoice', $data)
                ->setPaper('a4', 'portrait')
                ->setOption('margin-top', 10)
                ->setOption('margin-bottom', 10)
                ->setOption('margin-left', 10)
                ->setOption('margin-right', 10);

            // Use download method with proper filename
            return $pdf->download("invoice-{$invoiceNumber}.pdf");

        } catch (\Exception $e) {
            // If there's an error, redirect back with error message
            return redirect()->back()->with('error', 'Failed to generate invoice: ' . $e->getMessage());
        }
    }

    /**
     * Generate and download packing list PDF for an order
     */
    public function generatePackingList(Order $order)
    {
        try {
            // Load all necessary relationships
            $order->load([
                'buyer',
                'receiver',
                'transport',
                'batches.product'
            ]);

            // Calculate total weight and units
            $totalNetWeight = $order->batches->sum('net_weight');
            $totalGrossWeight = $order->batches->sum('gross_weight');
            $totalUnits = $order->batches->sum(function($batch) {
                return $batch->pivot->quantity;
            });
            $totalPackages = $order->batches->count();

            // Generate packing list number
            $packingListNumber = 'PL-' . date('Y') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            // Prepare data for PDF
            $data = [
                'order' => $order,
                'packingListNumber' => $packingListNumber,
                'packingDate' => now()->format('Y-m-d'),
                'totalNetWeight' => number_format($totalNetWeight, 2),
                'totalGrossWeight' => number_format($totalGrossWeight, 2),
                'totalUnits' => $totalUnits,
                'totalPackages' => $totalPackages,
            ];

            // Generate PDF
            $pdf = Pdf::loadView('pdfs.packing-list', $data)
                ->setPaper('a4', 'portrait')
                ->setOption('margin-top', 10)
                ->setOption('margin-bottom', 10)
                ->setOption('margin-left', 10)
                ->setOption('margin-right', 10);

            // Download PDF
            return $pdf->download("packing-list-{$packingListNumber}.pdf");

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to generate packing list: ' . $e->getMessage());
        }
    }

    /**
     * Preview invoice in browser (optional)
     */
    public function previewInvoice(Order $order)
    {
        try {
            $order->load(['buyer', 'receiver', 'transport', 'payment', 'batches.product']);

            $subtotal = $order->batches->sum(function($batch) {
                return $batch->pivot->quantity * $batch->pivot->price_per_unit;
            });

            $taxRate = 18;
            $taxAmount = $subtotal * ($taxRate / 100);
            $totalAmount = $subtotal + $taxAmount;

            $invoiceNumber = 'INV-' . date('Y') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            $data = [
                'order' => $order,
                'invoiceNumber' => $invoiceNumber,
                'invoiceDate' => now()->format('Y-m-d'),
                'dueDate' => now()->addDays(30)->format('Y-m-d'),
                'subtotal' => number_format($subtotal, 2),
                'taxRate' => $taxRate,
                'taxAmount' => number_format($taxAmount, 2),
                'totalAmount' => number_format($totalAmount, 2),
                'currency' => $order->payment->currency ?? 'USD',
            ];

            $pdf = Pdf::loadView('pdfs.invoice', $data);
            return $pdf->stream("invoice-{$invoiceNumber}.pdf");

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to preview invoice: ' . $e->getMessage());
        }
    }

    /**
     * Preview packing list in browser (optional)
     */
    public function previewPackingList(Order $order)
    {
        try {
            $order->load(['buyer', 'receiver', 'transport', 'batches.product']);

            $totalNetWeight = $order->batches->sum('net_weight');
            $totalGrossWeight = $order->batches->sum('gross_weight');
            $totalUnits = $order->batches->sum(function($batch) {
                return $batch->pivot->quantity;
            });

            $packingListNumber = 'PL-' . date('Y') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            $data = [
                'order' => $order,
                'packingListNumber' => $packingListNumber,
                'packingDate' => now()->format('Y-m-d'),
                'totalNetWeight' => number_format($totalNetWeight, 2),
                'totalGrossWeight' => number_format($totalGrossWeight, 2),
                'totalUnits' => $totalUnits,
                'totalPackages' => $order->batches->count(),
            ];

            $pdf = Pdf::loadView('pdfs.packing-list', $data);
            return $pdf->stream("packing-list-{$packingListNumber}.pdf");

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to preview packing list: ' . $e->getMessage());
        }
    }
}
