<?php

namespace Database\Factories;

use App\Models\Order;
use App\Models\Payment;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Payment>
 */
class PaymentFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $currencies = ['USD', 'EUR', 'RUB', 'CNY', 'MKD'];
        $paymentMethods = ['credit_card', 'debit_card', 'bank_transfer', 'paypal', 'stripe', 'cash'];
        $paymentStatuses = ['pending', 'processing', 'completed', 'failed', 'cancelled', 'refunded'];

        $dueDate = $this->faker->dateTimeBetween('-30 days', '+60 days');
        $currency = $this->faker->randomElement($currencies);

        // Exchange rates to MKD (Macedonian Denar)
        $exchangeRates = [
            'USD' => 57.50,   // 1 USD ≈ 57.50 MKD
            'EUR' => 61.50,   // 1 EUR ≈ 61.50 MKD
            'RUB' => 0.62,    // 1 RUB ≈ 0.62 MKD
            'CNY' => 8.00,    // 1 CNY ≈ 8.00 MKD
            'MKD' => 1.0000,  // 1 MKD = 1 MKD
        ];

        return [
            'amount' => $this->faker->randomFloat(2, 10, 5000),
            'currency' => $currency,
            'due_date' => $dueDate,
            'exchange_rate' => $exchangeRates[$currency] ?? $this->faker->randomFloat(4, 1, 100),
            'payment_date' => $this->faker->optional(0.7)->dateTimeBetween(
                min($dueDate, new \DateTime('now')),
                max($dueDate, new \DateTime('now'))
            ),
            'payment_method' => $this->faker->randomElement($paymentMethods),
            'payment_status' => $this->faker->randomElement($paymentStatuses),
            'order_id' => Order::factory(),
        ];
    }

    /**
     * Indicate that the payment is completed.
     */
    public function completed(): static
    {
        return $this->state(fn (array $attributes) => [
            'payment_status' => 'completed',
            'payment_date' => $this->faker->dateTimeBetween(
                min($attributes['due_date'], new \DateTime('now')),
                'now'
            ),
        ]);
    }

    /**
     * Indicate that the payment is pending.
     */
    public function pending(): static
    {
        return $this->state(fn (array $attributes) => [
            'payment_status' => 'pending',
            'payment_date' => null,
        ]);
    }

    /**
     * Indicate that the payment is overdue.
     */
    public function overdue(): static
    {
        return $this->state(fn (array $attributes) => [
            'payment_status' => 'pending',
            'due_date' => $this->faker->dateTimeBetween('-30 days', '-1 day'),
            'payment_date' => null,
        ]);
    }

    /**
     * Indicate that the payment is failed.
     */
    public function failed(): static
    {
        return $this->state(fn (array $attributes) => [
            'payment_status' => 'failed',
            'payment_date' => $this->faker->dateTimeBetween($attributes['due_date'], 'now'),
        ]);
    }

    /**
     * Set a specific currency with an appropriate exchange rate to MKD.
     */
    public function currency(string $currency): static
    {
        $exchangeRates = [
            'USD' => 57.50,   // 1 USD ≈ 57.50 MKD
            'EUR' => 61.50,   // 1 EUR ≈ 61.50 MKD
            'RUB' => 0.62,    // 1 RUB ≈ 0.62 MKD
            'CNY' => 8.00,    // 1 CNY ≈ 8.00 MKD
            'MKD' => 1.0000,  // 1 MKD = 1 MKD
        ];

        return $this->state(fn (array $attributes) => [
            'currency' => $currency,
            'exchange_rate' => $exchangeRates[$currency] ?? $this->faker->randomFloat(4, 1, 100),
        ]);
    }

    /**
     * Set a specific amount range.
     */
    public function amount(float $min = 10, float $max = 5000): static
    {
        return $this->state(fn (array $attributes) => [
            'amount' => $this->faker->randomFloat(2, $min, $max),
        ]);
    }

    /**
     * Associate with an existing order.
     */
    public function forOrder(Order $order): static
    {
        return $this->state(fn (array $attributes) => [
            'order_id' => $order->id,
        ]);
    }
}
