<?php

namespace Database\Seeders;

use App\Models\Payment;
use App\Models\Order;
use Illuminate\Database\Seeder;
use Carbon\Carbon;
use Illuminate\Support\Str;

class PaymentSeeder extends Seeder
{
    private $currencies = [
        'USD' => 1.00,    // Base currency
        'EUR' => 0.85,
        'MKD' => 55.50,
        'GBP' => 0.73,
        'JPY' => 110.25,
        'CNY' => 6.45,    // Added Chinese Yuan
        'RUB' => 75.80,   // Added Russian Ruble
    ];

    private $methods = ['bank_transfer', 'credit_card', 'cash', 'check', 'wire_transfer'];
    private $statuses = ['pending', 'paid', 'failed', 'refunded', 'cancelled'];

    public function run(): void
    {
        $orders = Order::all();

        foreach ($orders as $order) {
            $dueDate = Carbon::parse($order->date)->addDays(30);

            // Select a random currency and get its fixed exchange rate
            $currency = array_rand($this->currencies);
            $exchangeRate = $this->currencies[$currency];

            // Cap the amount to fit within decimal(10,2) which max is 9999999.99
            $amount = min($order->total_amount, 9999999.99);

            // Determine status
            $status = $this->statuses[array_rand($this->statuses)];

            // Set payment_date based on status
            if ($status === 'paid') {
                // For paid, set a random date between order date and due date
                $daysAfterOrder = rand(1, 30);
                $paymentDate = Carbon::parse($order->date)->addDays($daysAfterOrder);
            } else {
                // For non-paid, set to the due date
                $paymentDate = $dueDate;
            }

            Payment::create([
                'id' => (string) Str::uuid(),
                'order_id' => $order->id,
                'amount' => $amount,
                'currency' => $currency,
                'due_date' => $dueDate->format('Y-m-d'),
                'exchange_rate' => $exchangeRate,
                'payment_date' => $paymentDate->format('Y-m-d'),
                'payment_method' => $this->methods[array_rand($this->methods)],
                'payment_status' => $status,
            ]);
        }
    }
}
