namespace backend.Services;

using backend.Data;
using backend.DTOs;
using backend.Entities;
using backend.Helpers;
using backend.Models;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Options;
using Microsoft.IdentityModel.Tokens;
using System.IdentityModel.Tokens.Jwt;
using System.Security.Claims;

public interface IUserService
{
    Task<AuthenticateResponse> Authenticate(AuthenticateRequest model);
    Task<AuthenticateResponse> Register(CreateUserRequest req);
    Task<User> GetById(int id);
}

public class UserService : IUserService
{
    private readonly AppSettings _appSettings;
    private readonly DataContext _context = null;

    public UserService(IOptions<AppSettings> appSettings, DataContext context)
    {
        _appSettings = appSettings.Value;
        _context = context;
    }

    public async Task<AuthenticateResponse> Authenticate(AuthenticateRequest model)
    {
        User user = await _context.Users.FirstOrDefaultAsync(x => x.Email == model.Email && x.Password == model.Password);

        // return null if user not found
        if (user == null) return null;

        // authentication successful so generate jwt token
        var token = generateJwtToken(user);

        return new AuthenticateResponse { Email = user.Email, Id = user.Id, Token = token};
    }

    public async Task<User> GetById(int id)
    {
        return await _context.Users.FindAsync(id);
    }

    public async Task<AuthenticateResponse> Register(CreateUserRequest req)
    {
        User user = new User() { Email = req.Email, Password = req.Password };
        await _context.Users.AddAsync(user);
        await _context.SaveChangesAsync();
        var token = generateJwtToken(user);
        return new AuthenticateResponse { Email = user.Email, Id = user.Id, Token = token };
    }

    private string generateJwtToken(User user)
    {
        // generate token that is valid for 7 days
        var tokenHandler = new JwtSecurityTokenHandler();
        var key = System.Text.Encoding.ASCII.GetBytes(_appSettings.Secret);
        var tokenDescriptor = new SecurityTokenDescriptor
        {
            Subject = new ClaimsIdentity(new[] { new Claim("id", user.Id.ToString()) }),
            Expires = DateTime.UtcNow.AddDays(7),
            SigningCredentials = new SigningCredentials(new SymmetricSecurityKey(key), SecurityAlgorithms.HmacSha256Signature)
        };
        var token = tokenHandler.CreateToken(tokenDescriptor);
        return tokenHandler.WriteToken(token);
    }
}