﻿using backend.Data;
using backend.DTOs;
using backend.Entities;
using Microsoft.EntityFrameworkCore;

namespace backend.Services
{
    public interface IReviewService
    {
        public Task AddReview(CreateReviewRequest review, int userId);
        public Task<List<ReviewResponse>> GetReviews(int page, int pageSize);
        public Task<int> GetCount();
    }
    public class ReviewService : IReviewService
    {
        private readonly DataContext _context = null;

        public ReviewService(DataContext context)
        {
            _context = context;
        }

        public async Task AddReview(CreateReviewRequest review, int userId)
        {
            var user = await _context.Users.FindAsync(userId);
            var res = await _context.Restoraunts.Include(x => x.Reviews).FirstOrDefaultAsync();
            if(res.Reviews == null)
            {
                res.Reviews = new List<Review>();
            }
            res.Reviews.Add(new Review()
            {
                Title = review.Title,
                Description = review.Description,
                Stars = review.Stars,
                CreatedAt = DateTime.UtcNow,
                User = user
            });
            _context.Restoraunts.Update(res);
            await _context.SaveChangesAsync();
        }

        public async Task<int> GetCount()
        {
            return await _context.Reviews.CountAsync();
        }

        public async Task<List<ReviewResponse>> GetReviews(int page, int pageSize)
        {
            return await _context.Reviews
                .Include(x => x.User)
                .OrderByDescending(x => x.CreatedAt)
                .Skip(page * pageSize)
                .Take(pageSize)
                .Select(x => new ReviewResponse()
                {
                    CreatedAt = x.CreatedAt,
                    Description = x.Description,
                    Id = x.Id,
                    Stars = x.Stars,
                    Title = x.Title,
                    Username = x.User == null ? "Anonymous" : x.User.Email
                })
                .ToListAsync();
        }
    }
}
