﻿using backend.Data;
using backend.DTOs;
using backend.Entities;
using Microsoft.EntityFrameworkCore;

namespace backend.Services
{
    public interface IRestaurantService
    {
        public Task CreateRestaurant(string name, int userId);
        public Task<RestaurantResponse> GetRestaurant();
        public Task UploadImage(IFormFile file);
        public Task UpdateRestaurant(UpdateRestaurantRequest req);
    }
    public class RestaurantService : IRestaurantService
    {
        private readonly DataContext _context = null;

        public RestaurantService(DataContext context)
        {
            _context = context;
        }
        public async Task CreateRestaurant(string name, int userId)
        {
            User user = await _context.Users.FindAsync(userId);
            Restaurant restaurant = new Restaurant() { Name = name, Owner = user};
            await _context.Restoraunts.AddAsync(restaurant);
            await _context.SaveChangesAsync();
        }

        public async Task<RestaurantResponse> GetRestaurant()
        {
            RestaurantResponse res = await _context.Restoraunts
                .Include(x => x.Menu)
                .Include(x => x.Reviews).ThenInclude(x => x.User)
                .Select(x => new RestaurantResponse()
                {
                    Name = x.Name,
                    Address = x.Address,
                    Phone = x.Phone,
                    Base64Image = String.Format("data:image/png;base64,{0}", Convert.ToBase64String(x.Image)),
                    Menu = x.Menu.Select(x => new MenuItemResponse()
                    {
                        Id = x.Id,
                        Title = x.Title,
                        Description = x.Description,
                        Price = x.Price
                    }).ToList(),
                    Reviews = x.Reviews.Select(x => new ReviewResponse()
                    {
                        CreatedAt = x.CreatedAt,
                        Description = x.Description,
                        Id = x.Id,
                        Stars = x.Stars,
                        Title = x.Title,
                        Username = x.User == null ? "Anonymous" : x.User.Email
                    }).ToList(),
                    AverageReview = x.Reviews.Count>0 ? x.Reviews.Select(x => x.Stars).Average() : 0
                })
                .FirstOrDefaultAsync();
            return res;
        }

        public async Task UpdateRestaurant(UpdateRestaurantRequest req)
        {
            var restaurant = await _context.Restoraunts.FirstOrDefaultAsync();
            restaurant.Name = req.Name;
            restaurant.Address = req.Address;
            restaurant.Phone = req.Phone;
            _context.Restoraunts.Update(restaurant);
            await _context.SaveChangesAsync();
        }

        public async Task UploadImage(IFormFile file)
        {
            using (var memoryStream = new MemoryStream())
            {
                await file.CopyToAsync(memoryStream);
                var restaurant = await _context.Restoraunts.FirstOrDefaultAsync();
                restaurant.Image = memoryStream.ToArray();
                _context.Restoraunts.Update(restaurant);
                _context.SaveChanges();
            }
        }
    }
}
