﻿namespace backend.Services;

using backend.Data;
using backend.DTOs;
using backend.Email;
using backend.Entities;
using backend.Helpers;
using backend.Models;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Options;
using Microsoft.IdentityModel.Tokens;
using System.IdentityModel.Tokens.Jwt;
using System.Security.Claims;
using System.Security.Cryptography;
using System.Text;

public interface IUserService
{
    Task<AuthenticateResponse> Authenticate(AuthenticateRequest model);
    Task<AuthenticateResponse> Register(CreateUserRequest req, bool isFirst);
    Task<User> GetById(int id);
    Task SendEmailConfirmation(string email);
    Task SendPasswordReset(string email);
    Task ConfirmEmail(User user, string checkValid);
    Task ResetPassword(string checkValid, string password);
}

public class UserService : IUserService
{
    private readonly AppSettings _appSettings;
    private readonly DataContext _context = null;
    private readonly IEmailSender _emailSender;

    public UserService(IOptions<AppSettings> appSettings, DataContext context, IEmailSender emailSender)
    {
        _appSettings = appSettings.Value;
        _context = context;
        _emailSender = emailSender;
    }

    public async Task<AuthenticateResponse> Authenticate(AuthenticateRequest model)
    {
        User user = await _context.Users.FirstOrDefaultAsync(x => x.Email == model.Email && x.Password == model.Password);

        // return null if user not found
        if (user == null) return null;

        // authentication successful so generate jwt token
        var token = generateJwtToken(user);

        return new AuthenticateResponse { Email = user.Email, Id = user.Id, Token = token, IsAdmin = user.IsAdmin, IsConfirmed = user.IsConfirmed};
    }

    public async Task ConfirmEmail(User user, string checkValid)
    {
        if(user.ConfirmationURL != checkValid)
        {
            throw new Exception("Invalid check");
        }
        if(user.ConfirmationValidTo < DateTime.UtcNow)
        {
            throw new Exception("Link expired");
        }

        user.IsConfirmed = true;
        _context.Users.Update(user);
        await _context.SaveChangesAsync();
    }

    public async Task<User> GetById(int id)
    {
        return await _context.Users.FindAsync(id);
    }

    public async Task<AuthenticateResponse> Register(CreateUserRequest req, bool isFirst)
    {
        User user = new User() { Email = req.Email, Password = req.Password, IsAdmin = isFirst, IsConfirmed = false };
        await _context.Users.AddAsync(user);
        await _context.SaveChangesAsync();
        var token = generateJwtToken(user);
        return new AuthenticateResponse { Email = user.Email, Id = user.Id, Token = token, IsAdmin = user.IsAdmin, IsConfirmed = false };
    }

    public async Task ResetPassword(string checkValid, string password)
    {
        var user = await _context.Users.Where(x => x.PasswordResetURL == checkValid).FirstOrDefaultAsync();
        if (user == null)
        {
            throw new Exception("Invalid check");
        }
        if (user.PasswordResetValidTo < DateTime.UtcNow)
        {
            throw new Exception("Link expired");
        }

        user.Password = password;
        _context.Users.Update(user);
        await _context.SaveChangesAsync();
    }

    public async Task SendEmailConfirmation(string email)
    {
        User user = await _context.Users.FirstOrDefaultAsync(x => x.Email == email);
        user.ConfirmationURL = Guid.NewGuid().ToString();
        user.ConfirmationValidTo = DateTime.UtcNow.AddHours(24);
        _context.Users.Update(user);
        await _context.SaveChangesAsync();
        await _emailSender.SendEmailAsync(
            "Потврдете го вашиот емаил",
            "Ве молиме кликнете на следниот линк за да го потврдите вашиот емаил: http://localhost:3000/confirm?id=" + user.ConfirmationURL,
            email);
    }

    public async Task SendPasswordReset(string email)
    {
        User user = await _context.Users.FirstOrDefaultAsync(x => x.Email == email);
        user.PasswordResetURL = Guid.NewGuid().ToString();
        user.PasswordResetValidTo = DateTime.UtcNow.AddHours(24);
        _context.Users.Update(user);
        await _context.SaveChangesAsync();
        await _emailSender.SendEmailAsync(
           "Ресетирајте ја лозинката",
           "Ве молиме кликнете на следниот линк за да ја ресетирате лозинката: http://localhost:3000/reset?id=" + user.PasswordResetURL,
           email);
    }

    private string generateJwtToken(User user)
    {
        // generate token that is valid for 7 days
        var tokenHandler = new JwtSecurityTokenHandler();
        var key = System.Text.Encoding.ASCII.GetBytes(_appSettings.Secret);
        var tokenDescriptor = new SecurityTokenDescriptor
        {
            Subject = new ClaimsIdentity(new[] { new Claim("id", user.Id.ToString()) }),
            Expires = DateTime.UtcNow.AddDays(7),
            SigningCredentials = new SigningCredentials(new SymmetricSecurityKey(key), SecurityAlgorithms.HmacSha256Signature)
        };
        var token = tokenHandler.CreateToken(tokenDescriptor);
        return tokenHandler.WriteToken(token);
    }
    
    private string sha256Hash(String value)
    {
        using (SHA256 hash = SHA256.Create())
        {
            return String.Concat(hash
              .ComputeHash(Encoding.UTF8.GetBytes(value))
              .Select(item => item.ToString("x2")));
        }
    }
}