﻿using backend.Data;
using backend.DTOs;
using backend.Entities;
using Microsoft.EntityFrameworkCore;

namespace backend.Services
{
    public interface IToDoService
    {
        Task AddNewToDoItem(CreateToDoItemRequest item);
        Task<List<ToDoItemResponse>> GetToDoItems(string sortBy);
        Task UpdateToDoItemStatus(int id, ToDoStatus status);
    }
    public class ToDoService : IToDoService
    {
        private readonly DataContext _context = null;
        public ToDoService(DataContext context)
        {
            _context = context;
        }
        public async Task AddNewToDoItem(CreateToDoItemRequest item)
        {

            var linkedReview = await _context.Reviews.FirstOrDefaultAsync(x => x.Id == item.ReviewId);

            var newItem = new ToDoItem()
            {
                Description = item.Description,
                CreatedAt = DateTime.UtcNow,
                Status = ToDoStatus.New,
                Title = item.Title,
                LinkedReview = linkedReview,
                Priority = item.Priority
            };
            await _context.ToDoItems.AddAsync(newItem);
            await _context.SaveChangesAsync();
        }

        public async Task<List<ToDoItemResponse>> GetToDoItems(string sortBy)
        {
            var items = await _context.ToDoItems
                .Include(x => x.LinkedReview)
                .Select(x => new ToDoItemResponse()
                {
                    Description = x.Description,
                    CreatedAt = x.CreatedAt,
                    Id = x.Id,
                    LinkedReview = x.LinkedReview == null ? null : new ReviewResponse()
                    {
                        Description = x.LinkedReview.Description,
                        CreatedAt = x.LinkedReview.CreatedAt,
                        Id = x.LinkedReview.Id,
                        Stars = x.LinkedReview.Stars,
                        Title = x.LinkedReview.Title
                    },
                    Status = x.Status,
                    Title = x.Title,
                    Priority = x.Priority
                })
                .ToListAsync();

            if (sortBy.Equals("date"))
            {
                items = items.OrderByDescending(x => x.Status == ToDoStatus.New || x.Status == ToDoStatus.InProgress).ThenByDescending(x => x.CreatedAt).ToList();
            }
            else
            {
                items = items.OrderByDescending(x => x.Status == ToDoStatus.New || x.Status == ToDoStatus.InProgress).ThenBy(x => x.Priority).ToList();
            }

            return items;
        }

        public async Task UpdateToDoItemStatus(int id, ToDoStatus status)
        {
            var item = await _context.ToDoItems.FirstOrDefaultAsync(x => x.Id == id);
            item.Status = status;
            _context.Update(item);
            await _context.SaveChangesAsync();
        }
    }
}
