package com.example.skychasemk.services;

import com.example.skychasemk.dto.BookingDTO;
import com.example.skychasemk.enums.PaymentMethod;
import com.example.skychasemk.enums.PaymentStatus;
import com.example.skychasemk.model.*;
import com.example.skychasemk.repository.*;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
@Service
public class BookingService {

    private final BookingRepository bookingRepository;
    private final BookingFlightRepository bookingFlightRepository;
    private final FlightRepository flightRepository;
    private final PaymentRepository paymentRepository;
    private final ApplicationUserRepository userRepository;

    @Autowired
    public BookingService(BookingRepository bookingRepository,
                          BookingFlightRepository bookingFlightRepository,
                          FlightRepository flightRepository,
                          PaymentRepository paymentRepository, ApplicationUserRepository userRepository) {
        this.bookingRepository = bookingRepository;
        this.bookingFlightRepository = bookingFlightRepository;
        this.flightRepository = flightRepository;
        this.paymentRepository = paymentRepository;
        this.userRepository = userRepository;
    }

    @Transactional
    public Booking createBooking(BookingDTO dto) {
        if (dto.getUser() == null || dto.getFlights() == null || dto.getFlights().isEmpty()) {
            throw new IllegalArgumentException("User and flight IDs must not be null or empty");
        }
        ApplicationUser user = userRepository.findById(dto.getUser())
                .orElseThrow(() -> new RuntimeException("User not found with ID " + dto.getUser()));

        Booking booking = new Booking();
        booking.setUser(user);
        booking.setTotalCost(dto.getTotalCost());
        booking.setBookingDate(LocalDateTime.now());
        booking.setPaymentStatus(PaymentStatus.Processing);

        Booking savedBooking = bookingRepository.save(booking);

        for (Integer flightId : dto.getFlights()) {
            Flight flight = flightRepository.findById(flightId)
                    .orElseThrow(() -> new RuntimeException("Flight not found: " + flightId));

            if (flight.getAvailableSeats() <= 0) {
                throw new RuntimeException("No seats available for flight: " + flight.getFlightNumber());
            }

            flight.setAvailableSeats(flight.getAvailableSeats() - 1);
            flightRepository.save(flight);

            BookingFlight bf = new BookingFlight();
            bf.setId(new BookingFlightId(savedBooking.getBookingID(), flight.getFlightID()));
            bf.setBooking(savedBooking);
            bf.setFlight(flight);
            bookingFlightRepository.save(bf);
        }

        Payment payment = new Payment();
        payment.setBooking(savedBooking);
        payment.setUser(user);
        payment.setAmount(dto.getTotalCost());
        payment.setPaymentStatus(PaymentStatus.Processing);
        payment.setPaymentMethod(PaymentMethod.Credit);
        payment.setTransactionDate(LocalDateTime.now());

        paymentRepository.save(payment);

        savedBooking.setPaymentStatus(PaymentStatus.Processing);
        return bookingRepository.save(savedBooking);
    }
}
