package com.example.skychasemk.services;

import com.example.skychasemk.dto.PaymentDTO;
import com.example.skychasemk.enums.PaymentMethod;
import com.example.skychasemk.enums.PaymentStatus;
import com.example.skychasemk.model.ApplicationUser;
import com.example.skychasemk.model.Booking;
import com.example.skychasemk.model.Payment;
import com.example.skychasemk.repository.ApplicationUserRepository;
import com.example.skychasemk.repository.BookingRepository;
import com.example.skychasemk.repository.PaymentRepository;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;
@Service
public class PaymentService {

    @Autowired
    private PaymentRepository paymentRepository;

    @Autowired
    private ApplicationUserRepository userRepository;

    @Autowired
    private BookingRepository bookingRepository;

    public List<Payment> getAllPayments() {
        return paymentRepository.findAll();
    }

    public Payment getPaymentById(Integer id) {
        return paymentRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("Payment not found with ID " + id));
    }

    @Transactional
    public Payment createTransaction(PaymentDTO dto) {
        Booking booking = bookingRepository.findById(Math.toIntExact(dto.getBookingId()))
                .orElseThrow(() -> new RuntimeException("Booking not found with ID " + dto.getBookingId()));

        Optional<Payment> existingPayment = paymentRepository.findByBooking(booking);
        if (existingPayment.isPresent()) {
            throw new RuntimeException("Payment already exists for this booking");
        }

        ApplicationUser user = userRepository.findById(dto.getUserId())
                .orElseThrow(() -> new RuntimeException("User not found with ID " + dto.getUserId()));

        Payment payment = new Payment();
        payment.setBooking(booking);
        payment.setUser(user);
        payment.setAmount(dto.getAmount());
        payment.setPaymentStatus(PaymentStatus.Success);
        payment.setPaymentMethod(PaymentMethod.Credit);
        payment.setTransactionDate(LocalDateTime.now());

        booking.setPaymentStatus(PaymentStatus.Success);
        bookingRepository.save(booking);

        return paymentRepository.save(payment);
    }

    @Transactional
    public Payment updatePayment(Integer id, Payment paymentDetails) {
        Payment existing = getPaymentById(id);

        existing.setAmount(paymentDetails.getAmount());
        existing.setPaymentMethod(paymentDetails.getPaymentMethod());
        existing.setPaymentStatus(paymentDetails.getPaymentStatus());
        existing.setTransactionDate(paymentDetails.getTransactionDate());

        return paymentRepository.save(existing);
    }

    @Transactional
    public void deletePayment(Integer id) {
        if (!paymentRepository.existsById(id)) {
            throw new RuntimeException("Payment not found with ID " + id);
        }
        paymentRepository.deleteById(id);
    }
}
