package com.example.skychasemk.services;

import com.example.skychasemk.dto.DestinationDTO;
import com.example.skychasemk.model.Destination;
import com.example.skychasemk.repository.DestinationRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class DestinationService {

    @Autowired
    private DestinationRepository destinationRepository;

    private DestinationDTO toDTO(Destination destination) {
        return new DestinationDTO(
                destination.getDestinationId(),
                destination.getName(),
                destination.getCountry()
        );
    }

    private Destination toEntity(DestinationDTO dto) {
        Destination destination = new Destination();
        destination.setDestinationId(dto.getDestinationID());
        destination.setName(dto.getName());
        destination.setCountry(dto.getCountry());
        return destination;
    }

    public List<DestinationDTO> getAllDestinations() {
        return destinationRepository.findAll()
                .stream()
                .map(this::toDTO)
                .collect(Collectors.toList());
    }

    public DestinationDTO getDestinationById(Integer destinationId) {
        Destination destination = destinationRepository.findById(destinationId)
                .orElseThrow(() -> new RuntimeException("Destination not found with id " + destinationId));
        return toDTO(destination);
    }

    @Transactional
    public DestinationDTO createDestination(DestinationDTO dto) {
        Destination destination = toEntity(dto);
        Destination saved = destinationRepository.save(destination);
        return toDTO(saved);
    }

    @Transactional
    public DestinationDTO updateDestination(Integer destinationId, DestinationDTO dto) {
        Destination existing = destinationRepository.findById(destinationId)
                .orElseThrow(() -> new RuntimeException("Destination not found with id " + destinationId));

        existing.setName(dto.getName());
        existing.setCountry(dto.getCountry());

        Destination updated = destinationRepository.save(existing);
        return toDTO(updated);
    }

    @Transactional
    public void deleteDestination(Integer destinationId) {
        if (!destinationRepository.existsById(destinationId)) {
            throw new RuntimeException("Destination not found with id " + destinationId);
        }
        destinationRepository.deleteById(destinationId);
    }

    public List<String> getAllCities() {
        return destinationRepository.findAllCities();
    }
}
