package com.example.skychasemk.services;

import com.example.skychasemk.dto.ReviewDTO;
import com.example.skychasemk.enums.TargetType;
import com.example.skychasemk.model.ApplicationUser;
import com.example.skychasemk.model.Booking;
import com.example.skychasemk.model.Review;
import com.example.skychasemk.repository.ApplicationUserRepository;
import com.example.skychasemk.repository.BookingRepository;
import com.example.skychasemk.repository.ReviewRepository;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class ReviewService {

    @Autowired
    private ReviewRepository reviewRepository;

    @Autowired
    private ApplicationUserRepository userRepository;

    @Autowired
    private BookingRepository bookingRepository;


    public ReviewService(ReviewRepository reviewRepository) {
        this.reviewRepository = reviewRepository;
    }

    public List<Review> getReviewsByFlightId(Integer flightId) {
        return reviewRepository.findByFlightId(flightId);
    }

    public List<Review> getReviewsByAirportId(Integer airportId) {
        return reviewRepository.findByAirportId(airportId);
    }
    private ReviewDTO mapToDTO(Review review) {
        ReviewDTO dto = new ReviewDTO();
        dto.setReviewId(Long.valueOf(review.getReviewID()));
        dto.setReviewComment(review.getReviewComment());
        dto.setRating(review.getRating());
        dto.setTargetId(review.getTargetID());
        dto.setUserId(review.getUser().getUserId());
        return dto;
    }

    public List<ReviewDTO> getAllReviews() {
        return reviewRepository.findAll().stream()
                .map(this::mapToDTO)
                .collect(Collectors.toList());
    }

    public Review getReviewById(Integer id) {
        return reviewRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("Review not found with ID " + id));
    }

    @Transactional
    public Review submitReview(ReviewDTO dto) {
        ApplicationUser user = userRepository.findById(dto.getUserId())
                .orElseThrow(() -> new RuntimeException("User not found with ID " + dto.getUserId()));

        Review review = new Review();
        review.setUser(user);
        review.setTargetId(Long.valueOf(dto.getTargetId()));
        review.setTargetType(TargetType.Flight);
        review.setReviewComment(dto.getReviewComment());
        review.setRating(dto.getRating());
        review.setDate(LocalDateTime.now());

        return reviewRepository.save(review);
    }

    @Transactional
    public Review updateReview(Integer id, Review reviewDetails) {
        Review existing = getReviewById(id);

        existing.setReviewComment(reviewDetails.getReviewComment());
        existing.setRating(reviewDetails.getRating());
        existing.setDate(LocalDateTime.now());

        return reviewRepository.save(existing);
    }

    @Transactional
    public void deleteReview(Integer id) {
        if (!reviewRepository.existsById(id)) {
            throw new RuntimeException("Review not found with ID " + id);
        }
        reviewRepository.deleteById(id);
    }

    @Transactional
    public Review submitReview(Long userId, Long flightId, String reviewComment, int rating) {
        // Find all bookings for this user and flight
        List<Booking> bookings = bookingRepository.findBookingByUserIdAndFlightId(userId, flightId);

        if (bookings.isEmpty()) {
            throw new RuntimeException("No booking found for this flight and user");
        }

        Booking booking = bookings.stream()
                .max(Comparator.comparing(Booking::getBookingDate))
                .orElseThrow(() -> new RuntimeException("No booking found"));


        Review review = new Review();
        review.setUser(userRepository.findById(userId)
                .orElseThrow(() -> new RuntimeException("User not found")));
        review.setBooking(booking); // ← set booking ID to avoid null constraint
        review.setTargetId(flightId);
        review.setReviewComment(reviewComment);
        review.setRating(rating);
        review.setDate(LocalDateTime.now());

        return reviewRepository.save(review);
    }


}
