package com.example.skychasemk.services;

import com.example.skychasemk.dto.SupportTicketDTO;
import com.example.skychasemk.enums.TicketStatus;
import com.example.skychasemk.model.ApplicationUser;
import com.example.skychasemk.model.SupportTicket;
import com.example.skychasemk.repository.ApplicationUserRepository;
import com.example.skychasemk.repository.SupportTicketRepository;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class SupportTicketService {

    @Autowired
    private SupportTicketRepository supportTicketRepository;

    @Autowired
    private ApplicationUserRepository userRepository;

    public List<SupportTicket> getAllTickets() {
        return supportTicketRepository.findAll();
    }

    public List<SupportTicket> getResolvedTickets() {
        return supportTicketRepository.findResolvedTickets();
    }

    public SupportTicket getTicketById(Integer id) {
        return supportTicketRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("Ticket not found with ID " + id));
    }

    @Transactional
    public SupportTicket createTicket(SupportTicketDTO dto) {
        ApplicationUser user = userRepository.findById(dto.getUserId())
                .orElseThrow(() -> new RuntimeException("User not found with ID " + dto.getUserId()));

        SupportTicket ticket = new SupportTicket();
        ticket.setUser(user);
        ticket.setSubject(dto.getSubject());
        ticket.setDescription(dto.getDescription());
        ticket.setStatus(TicketStatus.Open);
        ticket.setDateCreated(LocalDateTime.now());

        return supportTicketRepository.save(ticket);
    }

    @Transactional
    public SupportTicket updateTicketStatus(Integer id, TicketStatus newStatus) {
        SupportTicket ticket = getTicketById(id);
        ticket.setStatus(newStatus);
        if (newStatus == TicketStatus.Resolved) {
            ticket.setDateResolved(LocalDateTime.now());
        }
        return supportTicketRepository.save(ticket);
    }

    @Transactional
    public void deleteTicket(Integer id) {
        if (!supportTicketRepository.existsById(id)) {
            throw new RuntimeException("Ticket not found with ID " + id);
        }
        supportTicketRepository.deleteById(id);
    }

    public SupportTicket updateTicket(Integer ticketId, TicketStatus newStatus) {
        return supportTicketRepository.findById(ticketId)
                .map(ticket -> {
                    ticket.setStatus(newStatus);
                    if (newStatus == TicketStatus.Resolved) {
                        ticket.setDateResolved(LocalDateTime.now());
                    }
                    return supportTicketRepository.save(ticket);
                })
                .orElseThrow(() -> new RuntimeException("Ticket not found with ID " + ticketId));
    }
}
