package com.example.skychasemk.services;

import com.example.skychasemk.dto.WishlistDTO;
import com.example.skychasemk.model.ApplicationUser;
import com.example.skychasemk.model.Flight;
import com.example.skychasemk.model.Wishlist;
import com.example.skychasemk.repository.ApplicationUserRepository;
import com.example.skychasemk.repository.FlightRepository;
import com.example.skychasemk.repository.WishlistRepository;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class WishlistService {

    @Autowired
    private WishlistRepository wishlistRepository;

    @Autowired
    private FlightRepository flightRepository;

    @Autowired
    private ApplicationUserRepository userRepository;

    public List<Wishlist> getAllWishlists() {
        return wishlistRepository.findAll();
    }

    public List<Wishlist> getWishlistForUser(Long userId) {
        return wishlistRepository.findWishlistByUserId(userId);
    }

    public Wishlist getWishlistItem(Integer wishlistId) {
        return wishlistRepository.findById(wishlistId)
                .orElseThrow(() -> new RuntimeException("Wishlist item not found with id " + wishlistId));
    }

    @Transactional
    public Wishlist addToWishlist(WishlistDTO dto) {
        ApplicationUser user = userRepository.findById(dto.getUserId())
                .orElseThrow(() -> new RuntimeException("User not found with id " + dto.getUserId()));

        // If target is a flight, verify flight exists (if you support destination targets, add similar handling)
        if (dto.getTargetId() != null && "Flight".equalsIgnoreCase(dto.getTargetType())) {
            Flight flight = flightRepository.findById(dto.getTargetId())
                    .orElseThrow(() -> new RuntimeException("Flight not found with id " + dto.getTargetId()));
            // check for existing
            Wishlist existing = wishlistRepository.findByTargetIdAndUserId(dto.getTargetId(), dto.getUserId());
            if (existing != null) {
                return existing; // already present
            }
            Wishlist item = new Wishlist();
            item.setUser(user);
            item.setTargetID(dto.getTargetId());
            item.setDateAdded(LocalDateTime.now());
            return wishlistRepository.save(item);
        } else {
            // If you support Destination or other target types, handle here
            throw new IllegalArgumentException("Unsupported or missing targetType. Expected 'Flight' (case-insensitive).");
        }
    }

    @Transactional
    public void removeFromWishlist(Integer userId, Integer targetId) {
        Wishlist existing = wishlistRepository.findByTargetIdAndUserId(targetId, Long.valueOf(userId));
        if (existing != null) {
            wishlistRepository.delete(existing);
        } else {
            // no-op or throw depending on desired behavior
        }
    }

    @Transactional
    public void deleteWishlistItem(Integer wishlistId) {
        if (!wishlistRepository.existsById(wishlistId)) {
            throw new RuntimeException("Wishlist item not found with id " + wishlistId);
        }
        wishlistRepository.deleteById(wishlistId);
    }
}
