<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\NewUserRequest;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use App\Notifications\WelcomeUser;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Propaganistas\LaravelPhone\PhoneNumber;

class UsersController extends Controller
{
    public function index()
    {
        return view("dashboard.users.index")->with([
            "users" => User::all()
        ]);
    }

    public function create()
    {
        return view("dashboard.users.create")->with([
            "roles" => Role::all(),
            "countries" => country()->all()
        ]);
    }

    public function store(NewUserRequest $request)
    {
        $user = new User();

        $user->name = $request->name;
        $user->surname = $request->surname;
        $user->email = $request->email;
        $user->country_code = $request->mobile_number_country;
        $user->mobile_number = PhoneNumber::make($request->mobile_number, $request->mobile_number_country)->formatInternational();
        $user->username = $request->username;
        $user->password = $user->generateTemporaryPassword();
        $user->security_code = $user->generateSecurityCode();
        $user->verify_token = $user->generateVerifyToken();

        $user->role_id = $request->userRole;

        $user->save();

        $user->userProfile()->create([
            "profile_link" => $user->generateProfileLink($request->name, $request->surname),
            "technoblog_email" => $user->generateTechnoblogEmail($request->name, $request->surname)
        ]);

        $user->notify(new WelcomeUser($user));

        Alert::flash("New user added successfully");

        return redirect()->route("dashboard.users.create");
    }

    public function block(Request $request, $id)
    {
        $user = User::find($id);
        $user->is_active = false;
        $user->save();
        Alert::flash($user->name . " blocked successfully");
        return redirect()->route("dashboard.users.index");
    }

    public function unblock(Request $request, $id)
    {
        $user = User::find($id);
        $user->is_active = true;
        $user->save();
        Alert::flash($user->name . " unblocked successfully");
        return redirect()->route("dashboard.users.index");
    }

    public function destroy(Request $request, $id)
    {
        $user = User::find($id);

        if (Storage::disk("uploads")->exists("/users/" . $user->userProfile->profile_photo_link)) {
            Storage::disk("uploads")->delete("/users/" . $user->userProfile->profile_photo_link);
        }

        if (Storage::disk("uploads")->exists("/users/" . $user->userProfile->cover_photo_link)) {
            Storage::disk("uploads")->delete("/users/" . $user->userProfile->cover_photo_link);
        }

        $user->userProfile->delete();
        $user->delete();

        Alert::flash($user->name . " deleted successfully");

        return redirect()->route("dashboard.users.index");
    }
}
