<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\EmailSettingsRequest;
use App\Http\Requests\Dashboard\PasswordSettingsRequest;
use App\Http\Requests\Dashboard\PhaseRequest;
use App\Http\Requests\Dashboard\PhotosSettingsRequest;
use App\Http\Requests\Dashboard\PostSecurityRequest;
use App\Http\Requests\Dashboard\SocialLinksSettingsRequest;
use App\Http\Requests\Dashboard\UsernameSettingsRequest;
use App\Http\Requests\Dashboard\UserProfileSettingsRequest;
use App\Models\Phase;
use App\Models\Post;
use App\Models\PostSecurity;
use App\Models\Role;
use App\Models\User;
use App\Models\UserProfile;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use App\Notifications\VerifyNewEmail;
use Illuminate\Support\Facades\Storage;
use Propaganistas\LaravelPhone\PhoneNumber;
use Propaganistas\LaravelIntl\Facades\Country;

class SettingsController extends Controller
{
    public function settings()
    {
        return view("dashboard.settings.index")->with([
            "userProfile" => auth()->user()->userProfile,
            "countries" => Country::all(),
            "adminAndEditors" => User::where("role_id", 1)->orWhere("role_id", 2)->get(),
            "userRoles" => Role::all(),
            "phases" => Phase::all(),
            "postSecurities" => PostSecurity::all()
        ]);
    }

    public function updatePersonalInformation(UserProfileSettingsRequest $request)
    {
        $userProfile = auth()->user()->userProfile;

        $userProfile->user->name = $request->name;
        $userProfile->user->surname = $request->surname;
        $userProfile->user->country_code = $request->mobile_number_country;
        $userProfile->user->mobile_number = PhoneNumber::make($request->mobile_number, $request->mobile_number_country)->formatInternational();

        $userProfile->profile_link = $request->profile_link;
        $userProfile->short_bio = $request->short_bio;
        $userProfile->technoblog_email = $userProfile->user->generateTechnoblogEmail($request->name, $request->surname);

        $userProfile->user->save();
        $userProfile->save();

        Alert::flash("Updated successfully");

        return redirect()->route("dashboard.settings.index");
    }

    public function updatePhotos(PhotosSettingsRequest $request)
    {
        $userProfile = auth()->user()->userProfile;
        $this->updatePhotosHelper($request, $userProfile);
        $userProfile->save();

        Alert::flash("Updated successfully");

        return redirect()->route("dashboard.settings.index");
    }

    public function updateSocialLinks(SocialLinksSettingsRequest $request)
    {
        $userProfile = auth()->user()->userProfile;

        $userProfile->facebook_link = $request->facebook;
        $userProfile->instagram_link = $request->instagram;
        $userProfile->twitter_link = $request->twitter;
        $userProfile->youtube_link = $request->youtube;
        $userProfile->skype_link = $request->skype;

        $userProfile->save();

        Alert::flash("Updated successfully");

        return redirect()->route("dashboard.settings.index");
    }

    public function updateUsername(UsernameSettingsRequest $request)
    {
        $user = auth()->user();
        $user->username = $request->username;
        $user->save();

        auth()->logout();
        session()->flush();

        return redirect()->route("auth.loginShow");
    }

    public function updatePassword(PasswordSettingsRequest $request)
    {
        $user = auth()->user();
        $user->password = bcrypt($request->password);
        $user->save();

        auth()->logout();
        session()->flush();

        return redirect()->route("auth.loginShow");
    }

    public function updateEmail(EmailSettingsRequest $request)
    {
        $user = auth()->user();

        $user->email = $request->email;
        $user->is_active = false;
        $user->security_code = $user->generateSecurityCode();
        $user->verify_token = $user->generateVerifyToken();

        $user->save();

        $user->notify(new VerifyNewEmail($user));

        auth()->logout();
        session()->flush();

        return redirect()->route("auth.loginShow");
    }

    public function transferPostsAndDeleteUser(Request $request)
    {
        $from = $request->from;
        $to = $request->to;

        if ($from == $to) {
            return redirect()->back()->withInput()->withErrors([
                "cantDelete" => "Can't transfer posts to same user"
            ]);
        }

        $posts = Post::where("user_id", $from)->get();
        $userFrom = User::find($from);
        $userTo = User::find($to);
        $selfAccount = false;

        if ($request->has("password")) {
            $selfAccount = true;
            if (!Hash::check($request->password, $userFrom->password)) {
                return redirect()->back()->withInput()->withErrors([
                    "cantDelete" => "Your password is invalid"
                ]);
            }
        }

        if ($posts != null) {
            foreach ($posts as $post) {
                $post->user()->associate($userTo);
                $post->confirmed_by = $userTo->id;
                $post->save();
            }
        }

        if (Storage::disk("uploads")->exists("/users/" . $userFrom->userProfile->profile_photo_link)) {
            Storage::disk("uploads")->delete("/users/" . $userFrom->userProfile->profile_photo_link);
        }

        if (Storage::disk("uploads")->exists("/users/" . $userFrom->userProfile->cover_photo_link)) {
            Storage::disk("uploads")->delete("/users/" . $userFrom->userProfile->cover_photo_link);
        }

        $userFrom->delete();
        $userFrom->userProfile->delete();

        if ($selfAccount) {
            return redirect()->route("blog.index");
        }

        Alert::flash($userFrom->name . " deleted successfully");

        return redirect()->route("dashboard.users.index");
    }

    public function phaseStore(PhaseRequest $request)
    {
        if ($request->has("phase_id") && !is_null($request->phase_id)) {
            $phase = Phase::findOrFail($request->phase_id);
        } else {
            $phase = new Phase();
        }

        $phase->name = $request->name;
        $phase->reviewer()->associate(Role::find($request->reviewer_role_id));
        $phase->user()->associate(auth()->user());
        $phase->save();

        if ($request->has("phase_id") && !is_null($request->phase_id)) {
            Alert::flash($phase->name . " updated successfully");
        } else {
            Alert::flash($phase->name . " created successfully");
        }

        return redirect()->route("dashboard.settings.index");
    }

    public function phaseDestroy($id)
    {
        $phase = Phase::findOrFail($id);
        $phaseName = $phase->name;
        $phase->delete();

        Alert::flash($phaseName . " deleted successfully");

        return redirect()->route("dashboard.settings.index");
    }

    public function postSecurityStore(PostSecurityRequest $request)
    {
        if ($request->has("post_security_id") && !is_null($request->post_security_id)) {
            $postSecurity = PostSecurity::findOrFail($request->post_security_id);
        } else {
            $postSecurity = new PostSecurity();
        }

        $postSecurity->role()->associate(Role::find($request->role_id_to_be_reviewed));
        $postSecurity->phase_ids = $request->phase_ids;

        $postSecurity->save();

        if ($request->has("post_security_id") && !is_null($request->post_security_id)) {
            Alert::flash("Post security updated successfully");
        } else {
            Alert::flash("Post security created successfully");
        }

        return redirect()->route("dashboard.settings.index");
    }

    public function postSecurityDestroy($id)
    {
        PostSecurity::findOrFail($id)->delete();
        Alert::flash("Post security deleted successfully");
        return redirect()->route("dashboard.settings.index");
    }

    private function updatePhotosHelper(Request $request, UserProfile $userProfile)
    {
        if ($request->hasFile("profile_image")) {

            $image = $request->file("profile_image");
            $extension = $image->getClientOriginalExtension();
            $imageName = $this->createImageName($extension);

            if (!empty($userProfile->profile_photo_link)) {
                Storage::disk('uploads')->delete("/users/" . $userProfile->profile_photo_link);
            }

            Storage::disk('uploads')->put("/users/" . $imageName, File::get($image));
            $userProfile->profile_photo_link = $imageName;
        }

        if ($request->hasFile("cover_image")) {

            $image = $request->file("cover_image");
            $extension = $image->getClientOriginalExtension();
            $imageName = $this->createImageName($extension);

            if (!empty($userProfile->cover_photo_link)) {
                Storage::disk('uploads')->delete("/users/" . $userProfile->cover_photo_link);
            }

            Storage::disk('uploads')->put("/users/" . $imageName, File::get($image));
            $userProfile->cover_photo_link = $imageName;
        }
    }

    private function createImageName($extension)
    {
        return auth()->user()->id . "-" . Str::random(10) . "." . $extension;
    }
}
