package com.tourMate.services.impl;

import com.tourMate.dao.RestaurantDao;
import com.tourMate.dto.RestaurantDto;
import com.tourMate.dto.RestaurantReservationDto;
import com.tourMate.dto.RestaurantReservationUserDto;
import com.tourMate.services.RestaurantManager;
import com.tourMate.entities.*;
import com.tourMate.services.UsersManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
public class RestaurantManagerImpl implements RestaurantManager {

    @Autowired
    RestaurantDao restaurantDao;
    @Autowired
    UsersManager usersManager;

    @Override
    public void createRestaurant(Restaurant restaurant, long userId) {
        User u = usersManager.findUserByID(userId);
        Restaurant r = new Restaurant(restaurant.getRestaurantName(), restaurant.getRestaurantLocation(), restaurant.getCousineType(), restaurant.getRestaurantDescription(), restaurant.getRestaurantEdbs(), u);
        restaurantDao.createRestaurant(r);
    }

    @Override
    public void deleteRestaurant(Long restaurantID) {
        Restaurant r = findRestaurantByID(restaurantID);
        restaurantDao.deleteRestaurant(r);
    }

    @Override
    public void addMenuToRestaurant(long restaurantId, Menu menu) {
        Restaurant r = findRestaurantByID(restaurantId);
        menu.setRestaurant(r);
        restaurantDao.addMenuToRestaurant(menu);
    }

    @Override
    public List<Restaurant> getRestaurantsByUser(long userId) {
        User u = usersManager.findUserByID(userId);
        return restaurantDao.getRestaurantsByUser(u);
    }

    @Override
    public void deleteRestaurant(long restaurantID) {
        Restaurant r = findRestaurantByID(restaurantID);
        restaurantDao.deleteRestaurant(r);
    }

    @Override
    public List<Restaurant> getRestaurants() {
        return restaurantDao.getRestaurants();
    }

    @Override
    public Restaurant findRestaurantByID(long restaurantID) {
        return restaurantDao.findRestaurantByID(restaurantID);
    }

    @Override
    public void addRestaurantImage(Long restaurant, List<String> url) {
        Restaurant r = restaurantDao.findRestaurantByID(restaurant);
        url.stream().map(file -> new RestaurantImages(r, file)).forEach(x -> restaurantDao.addRestaurantImage(x));
    }

    @Override
    public void removeRestaurantImage(long restaurantImageId) {
        RestaurantImages restaurantImage = findRestaurantImageById(restaurantImageId);
        restaurantDao.removeRestaurantImage(restaurantImage);
    }

    @Override
    public RestaurantImages findRestaurantImageById(long restaurantImageId) {
        return restaurantDao.findRestaurantImageById(restaurantImageId);
    }

    @Override
    public List<RestaurantImages> getRestaurantImages(long restaurantID) {
        return restaurantDao.getRestaurantImages(restaurantID);
    }

    @Override
    public List<RestaurantDto> getTablesByDateAndLocation(String restaurantLocation, Date hourFrom, Date hourTo, int noSeats) {
        List<RestaurantsAvailible> restaurantsAvailibles = restaurantDao.getTablesByDateAndLocation(restaurantLocation, hourFrom, hourTo, noSeats);
        Map<Restaurant, List<RestaurantsAvailible>> tablesByRestaurants = restaurantsAvailibles.stream().collect(Collectors.groupingBy(x -> x.getRestaurantTable().getRestaurant()));
        List<RestaurantDto> restaurantsList = tablesByRestaurants.keySet().stream()
                .map(x -> new RestaurantDto(
                        x.getRestaurantID(),
                        x.getRestaurantName(),
                        x.getRestaurantLocation(),
                        x.getCousineType(),
                        x.getRestaurantDescription(),
                        x.getRestaurantEdbs(),
                        x.getMenus(),
                        tablesByRestaurants.get(x),
                        getReviewsForRestaurant(x.getRestaurantID()),
                        getReviewsForRestaurant(x.getRestaurantID()).stream().mapToDouble(Reviews::getNumStar).average().orElse(0),
                        getRestaurantImages(x.getRestaurantID()),
                        getMenuImagesByRestaurant(x.getRestaurantID())
                )).toList();
        return restaurantsList;
    }

    @Override
    public void editRestaurant(long restaurantID, String restaurantName, String restaurantLocation, String cousineType, String restaurantDescription, String restaurantEdbs, User restaurantOwner) {
        Restaurant res = findRestaurantByID(restaurantID);
        res.setRestaurantName(restaurantName);
        res.setRestaurantLocation(restaurantLocation);
        res.setRestaurantEdbs(restaurantEdbs);
        res.setRestaurantDescription(restaurantDescription);
        res.setRestaurantOwner(restaurantOwner);
        res.setCousineType(cousineType);
        restaurantDao.createRestaurant(res);
    }

    @Override
    public void editReservation(long restaurantsTableId, RestaurantsTable rt, Date hourFrom, Date hourTo, int noSeats, User user){
        RestaurantReservations r = findReservationByID(restaurantsTableId);
        r.setTable(rt);
        r.setTimeFrom(hourFrom);
        r.setTimeTo(hourTo);
        r.setNoSeats(noSeats);
        r.setUser(user);
        restaurantDao.saveReservation(r);
    }

    @Override
    public List<RestaurantsTable> getRestaurantTables(long restaurantID) {
        return restaurantDao.getRestaurantTables(restaurantID);
    }

    @Override
    public RestaurantsTable findTableById(long tableId) {
        return restaurantDao.findTableById(tableId);
    }

    @Override
    public void createTable(Long restaurantId, int noSeats) {
        Restaurant r = restaurantDao.findRestaurantByID(restaurantId);
        RestaurantsTable restaurantsTable = new RestaurantsTable(r, noSeats);
        restaurantDao.saveTable(restaurantsTable);
    }

    @Override
    public void editTable(Restaurant restaurant, long tableId, int noSeats) {
        RestaurantsTable resTable = findTableById(tableId);
        resTable.setRestaurant(restaurant);
        resTable.setNoSeats(noSeats);
        restaurantDao.saveTable(resTable);
    }

    @Override
    public void deleteTable(long tableId) {
        RestaurantsTable rt = findTableById(tableId);
        restaurantDao.deleteTable(rt);
    }

    @Override
    public void createTableAvailable(Long rt, Date hourFrom, Date hourTo, int numTables) {
        RestaurantsTable rtabl = findTableById(rt);
        RestaurantsAvailible ra = new RestaurantsAvailible(rtabl, hourFrom, hourTo, numTables);
        restaurantDao.saveTableAvailable(ra);
    }

    @Override
    public List<RestaurantsAvailible> getTablesAvailabilityById(Long id) {
        return restaurantDao.getTablesAvailabilityById(id);
    }

    @Override
    public void editTableAvailable(long tableId, Restaurant restaurant, int noSeats) {
        RestaurantsTable rt = findTableById(tableId);
        rt.setRestaurant(restaurant);
        rt.setNoSeats(noSeats);
        restaurantDao.saveTable(rt);
    }

    @Override
    public void deleteTableAvailable(long tableId) {
        RestaurantsAvailible ra = findAvailableReservationByID(tableId);
        restaurantDao.deleteTableAvailable(ra);
    }

    @Override
    public RestaurantsAvailible findAvailableReservationByID(long tableId){
        return restaurantDao.findAvailableReservationByID(tableId);
    }

    @Override
    public void createReservation(RestaurantsTable rt, Date dateFrom, Date dateTo, User user) {
        RestaurantReservations restaurantReservations = new RestaurantReservations(rt, dateFrom, dateTo, rt.getNoSeats(), user);
        restaurantDao.createReservation(restaurantReservations);
    }


    @Override
    public void deleteReservation(long tableId) {
        RestaurantReservations r = findReservationByID(tableId);
        RestaurantsAvailible ra = new RestaurantsAvailible(r.getTable(), r.getTimeFrom(), r.getTimeTo(), r.getNoSeats());
        restaurantDao.saveTableAvailable(ra);
        restaurantDao.deleteReservation(r);
    }

    @Override
    public RestaurantReservations findReservationByID(long tableId) {
        return restaurantDao.findReservationByID(tableId);
        //return null;
    }

    @Override
    public List<RestaurantReservations> findReservationByUser(User user) {
        return restaurantDao.findReservationByUser(user);
    }

    @Override
    public List<RestaurantReservationDto> findReservationByRestaurant(Long restaurant) {
        Restaurant r = findRestaurantByID(restaurant);
        List<RestaurantReservations> res = restaurantDao.findReservationByRestaurant(r);
        return res.stream().map(x -> new RestaurantReservationDto(
                x.getUser(),
                x.getTable(),
                x.getTimeFrom(),
                x.getTimeTo()
        )).collect(Collectors.toList());
    }

    @Override
    public List<RestaurantReservations> getReservations() {
        return restaurantDao.getReservations();
    }

    @Override
    public List<Restaurant> searchByRestaurantName(String restaurantName) {
        return restaurantDao.searchByRestaurantName(restaurantName);
    }

    @Override
    public List<Restaurant> searchByRestaurantLocation(String restaurantLocation) {
        return restaurantDao.searchByRestaurantLocation(restaurantLocation);
    }

    @Override
    public List<RestaurantsTable> searchByNoSeats(int noSeats) {
        return restaurantDao.searchByNoSeats(noSeats);
    }

    @Override
    public List<Reviews> getReviewsForRestaurant(Long restaurantId) {
        Restaurant restaurant = findRestaurantByID(restaurantId);
        return restaurantDao.findReviewsByRestaurant(restaurant);
    }

    @Override
    public void createReservation(Long userId, Long restaurantTableId, Long restaurantAvailibleId, String hourFrom, String hourTo, Date date) {
        User u = usersManager.findUserByID(userId);
        RestaurantsTable restaurantTable = restaurantDao.findTableById(restaurantTableId);
        RestaurantsAvailible restaurantsAvailible = restaurantDao.findAvailableReservationByID(restaurantAvailibleId);
        Date dateFrom = date;
        Date dateTo = Date.from(date.toInstant());
        String[] splittedFrom = hourFrom.split(":");
        String[] splittedTo = hourTo.split(":");
        dateFrom.setHours(Integer.parseInt(splittedFrom[0]));
        dateFrom.setMinutes(Integer.parseInt(splittedFrom[1]));
        dateTo.setHours(Integer.parseInt(splittedTo[0]));
        dateTo.setMinutes(Integer.parseInt(splittedTo[1]));
        RestaurantReservations reservation = new RestaurantReservations(restaurantTable, dateFrom, dateTo, restaurantTable.getNoSeats(), u);
        restaurantDao.createReservation(reservation);
        editTableAvailability(restaurantsAvailible, restaurantTable, dateFrom, dateTo);
    }

    public void editTableAvailability(RestaurantsAvailible restaurantsAvailible, RestaurantsTable table, Date dateFrom, Date dateTo) {
        restaurantsAvailible.setNumTables(restaurantsAvailible.getNumTables() - 1);
        RestaurantsAvailible ra1 = new RestaurantsAvailible(table, restaurantsAvailible.getHourFrom(), dateFrom, 1);
        RestaurantsAvailible ra2 = new RestaurantsAvailible(table, dateTo, restaurantsAvailible.getHourTo(), 1);

        restaurantDao.saveTableAvailable(ra1);
        restaurantDao.saveTableAvailable(ra2);
    }

    @Override
    public List<RestaurantReservationUserDto> findValidRestaurantReservationsByUser(Long userId) {
        User u = usersManager.findUserByID(userId);
        List<RestaurantReservations> reservations = restaurantDao.findReservationByUser(u);
        return reservations.stream().map(x -> new RestaurantReservationUserDto(
                x.getReservationId(),
                x.getUser(),
                x.getTable(),
                x.getTimeFrom(),
                x.getTimeTo(),
                x.getNoSeats(),
                x.getTable().getRestaurant().getRestaurantName(),
                x.getTable().getRestaurant().getRestaurantLocation(),
                "",
                x.getTable().getRestaurant().getRestaurantID(),
                x.getReviewed()
        )).toList();
    }

    @Override
    public List<RestaurantReservationUserDto> findPastRestaurantReservationsByUser(Long userId) {
        User u = usersManager.findUserByID(userId);
        List<RestaurantReservations> reservations = restaurantDao.findPastReservationsByUser(u);
        return reservations.stream().map(x -> new RestaurantReservationUserDto(
                x.getReservationId(),
                x.getUser(),
                x.getTable(),
                x.getTimeFrom(),
                x.getTimeTo(),
                x.getNoSeats(),
                x.getTable().getRestaurant().getRestaurantName(),
                x.getTable().getRestaurant().getRestaurantLocation(),
                "",
                x.getTable().getRestaurant().getRestaurantID(),
                x.getReviewed()
        )).toList();
    }

    @Override
    public List<MenuImages> getMenuImages(Long id) {
        return restaurantDao.getMenuImages(id);
    }

    @Override
    public void addMenuImage(Long menuId, List<String> url) {
        Menu r = restaurantDao.findMenuId(menuId);
        url.stream().map(file -> new MenuImages(r, file)).forEach(x -> restaurantDao.addMenuImage(x));
    }

    @Override
    public Menu findMenuById(Long menuId) {
        return restaurantDao.findMenuId(menuId);
    }

    @Override
    public void setReservationReviewed(Long reservationId) {
        RestaurantReservations restaurantReservations = findReservationByID(reservationId);
        restaurantReservations.setReviewed(true);
        restaurantDao.saveReservation(restaurantReservations);
    }

    @Override
    public List<MenuImages> getMenuImagesByRestaurant(long restaurantId) {
        Restaurant r = findRestaurantByID(restaurantId);
        return restaurantDao.getMenuImagesByRestaurant(r);
    }
}
