package com.tourMate.dao.impl;

import com.tourMate.dao.RestaurantDao;
import com.tourMate.entities.*;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.transaction.Transactional;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service
public class RestaurantDaoImpl implements RestaurantDao {
    @PersistenceContext
    EntityManager em;

    @Override
    @Transactional
    public void createRestaurant(Restaurant restaurant) {
        em.persist(restaurant);
    }

    @Override
    public List<RestaurantsAvailible> getTablesByDateAndLocation(String restaurantLocation, Date hourFrom, Date hourTo, int noSeats){
        return em.createQuery("select hr from RestaurantsAvailible hr where hr.hourFrom <= :hourFrom and hr.hourTo >= :hourTo " +
                        "and hr.restaurantTable.restaurant.restaurantLocation LIKE :restaurantLocation and hr.restaurantTable.noSeats >= :noSeats")
                .setParameter("restaurantLocation", restaurantLocation)
                .setParameter("hourFrom", hourFrom)
                .setParameter("hourTo", hourTo)
                .setParameter("noSeats", noSeats)
                .getResultList();
    }

    @Transactional
    @Override
    public void addRestaurantImage(RestaurantImages restaurantImages){
        em.persist(restaurantImages);
    }

    @Transactional
    @Override
    public void removeRestaurantImage(RestaurantImages image){
        em.remove(image);
    }

    @Transactional
    @Override
    public RestaurantImages findRestaurantImageById(long restaurantImageId){
        return em.find(RestaurantImages.class, restaurantImageId);
    }

    @Override
    public void deleteRestaurant(Restaurant r) {
        em.remove(r);
    }

    @Override
    public List<Restaurant> getRestaurants() {
        return em.createQuery("select r from Restaurant r order by r.restaurantID").getResultList();
    }

    @Override
    @Transactional
    public void addMenuToRestaurant(Menu menu) {
        em.persist(menu);
    }

    @Override
    public Restaurant findRestaurantByID(long restaurantID) {
        return em.find(Restaurant.class, restaurantID);
    }

    @Override
    public List<RestaurantImages> getRestaurantImages(long restaurantID) {
        return em.createQuery("select ri from RestaurantImages ri where ri.restaurant.restaurantID = :restaurantID").setParameter("restaurantID", restaurantID).getResultList();
    }

    @Override
    public List<RestaurantsTable> getRestaurantTables(long restaurantID) {
        return em.createQuery("select rt from RestaurantsTable rt where rt.restaurant.restaurantID = :restaurantId").setParameter("restaurantId", restaurantID).getResultList();
    }

    @Override
    public RestaurantsTable findTableById(long tableId) {
        return em.find(RestaurantsTable.class, tableId);
    }

    @Override
    @Transactional
    public void saveTable(RestaurantsTable resTable) {
        em.persist(resTable);
    }


    @Override
    @Transactional
    public void deleteTable(RestaurantsTable rt) {
        em.persist(rt);
    }

    @Override
    public List<Restaurant> getRestaurantsByUser(User u) {
        return em.createQuery("select r from Restaurant r where r.restaurantOwner = :u or r.restaurantOwner in :users")
                .setParameter("u", u)
                .setParameter("users", u.getConnectedAccounts())
                .getResultList();
    }

    @Override
    public List<RestaurantsAvailible> getTablesAvailabilityById(Long id)
    {
        return em.createQuery("SELECT ra from RestaurantsAvailible ra WHERE ra.restaurantTable.id = :id").setParameter("id", id).getResultList();
    }

    @Override
    @Transactional
    public void saveTableAvailable(RestaurantsAvailible ra){
        em.persist(ra);
    }

    @Override
    @Transactional
    public void deleteTableAvailable(RestaurantsAvailible ra) {
        em.remove(ra);
    }

    @Override
    public List<RestaurantsAvailible> getTablesAvailability() {
        return em.createQuery("SELECT a from RestaurantsAvailible a ORDER BY a.restaurantAvailibleId").getResultList();
    }

    @Override
    @Transactional
    public void createReservation(RestaurantReservations reservations) {
        em.persist(reservations);
    }


    @Override
    @Transactional
    public void deleteReservation(RestaurantReservations r) {
        em.remove(r);
    }

    @Override
    public RestaurantReservations findReservationByID(long tableId) {
        return em.find(RestaurantReservations.class, tableId);
    }

    @Override
    public RestaurantsAvailible findAvailableReservationByID(long availibleId) {
        return em.find(RestaurantsAvailible.class, availibleId);
    }

    @Override
    public List<RestaurantReservations> findReservationByUser(User user) {
        return em.createQuery("SELECT r from RestaurantReservations r where r.user = :user and r.timeTo >= now()")
                .setParameter("user", user)
                .getResultList();
    }

    @Override
    public List<RestaurantReservations> findReservationByRestaurant(Restaurant restaurant) {
        return em.createQuery("select rr from RestaurantReservations rr where rr.table.restaurant = :rest").setParameter("rest", restaurant).getResultList();
    }

    @Override
    public List<RestaurantReservations> getReservations() {
        return em.createQuery("select r from Restaurant r order by r.restaurantName").getResultList();
    }

    @Override
    public List<Restaurant> searchByRestaurantName(String restaurantName) {
        return em.createQuery("select rt from RestaurantsTable rt where rt.restaurant.restaurantName = :restaurantName").setParameter("restaurantName", restaurantName).getResultList();
    }

    @Override
    public List<Restaurant> searchByRestaurantLocation(String restaurantLocation) {

        return em.createQuery("select rt from RestaurantsTable rt where rt.restaurant.restaurantLocation = :restaurantLocation").setParameter("restaurantLocation", restaurantLocation).getResultList();

    }

    @Override
    public List<RestaurantsTable> searchByNoSeats(int noSeats) {
        return em.createQuery("select rt from RestaurantsTable rt where rt.noSeats = noSeats").getResultList();
    }

    @Override
    public List<Reviews> findReviewsByRestaurant(Restaurant restaurant) {
        return em.createQuery("select r from Reviews r where r.restaurant = :restaurant")
                .setParameter("restaurant", restaurant)
                .getResultList();
    }

    @Override
    public List<RestaurantReservations> findPastReservationsByUser(User user) {
        return em.createQuery("SELECT r from RestaurantReservations r where r.user = :user and r.timeTo <= now()")
                .setParameter("user", user).getResultList();

    }

    @Override
    public List<MenuImages> getMenuImages(long menuId) {
        return em.createQuery("select ri from MenuImages ri where ri.menu.id = :menuId").setParameter("menuId", menuId).getResultList();
    }

    @Override
    public List<MenuImages> getMenuImagesByRestaurant(Restaurant resta) {
        return em.createQuery("select mi from MenuImages mi where mi.menu.restaurant = :restaurant")
                .setParameter("restaurant", resta)
                .getResultList();
    }

    @Transactional
    @Override
    public void addMenuImage(MenuImages menuImages) {
        em.persist(menuImages);
    }

    @Override
    public Menu findMenuId(Long menuId) {
        return em.find(Menu.class, menuId);
    }

    @Override
    @Transactional
    public void saveReservation(RestaurantReservations restaurantReservations) {
        em.persist(restaurantReservations);
    }
}