package com.tourMate.dao.impl;

import com.tourMate.dao.UsersDao;
import com.tourMate.entities.Role;
import com.tourMate.entities.User;
import com.tourMate.events.OnProfileEnabledEvent;
import com.tourMate.events.OnRegistrationSuccessEvent;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.transaction.Transactional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;

@Repository
public class UsersDaoImpl implements UsersDao {

    @PersistenceContext
    EntityManager em;
    @Autowired
    PasswordEncoder passwordEncoder;
    @Autowired
    private ApplicationEventPublisher eventPublisher;

    @Transactional
    @Override
    public void createUser(String name, String surname, String email, Date birthDate, String address, String contact) {
        Role r = em.find(Role.class, 2);
        User user = new User(name, surname, email, passwordEncoder.encode("password"), birthDate, address, contact, r);
        user.setProvider("local");
        user.setLocked(true);
        em.persist(user);
        if(user.getProvider().equals("local"))
        {
            eventPublisher.publishEvent(new OnRegistrationSuccessEvent(user));
        }
    }

    @Transactional
    @Override
    public void deleteUser(long userID) {
        User u = findUserByID(userID);
        em.remove(u);
    }

    @Override
    public List<User> getCreatedUsers() {
        return em.createQuery("from User order by userID").getResultList();
    }

    @Override
    public List<Role> getRoles() {
        return em.createQuery("SELECT r from Role r ORDER BY r.id").getResultList();
    }

    @Override
    public User findUserByID(long userID) {
        return em.find(User.class, userID);
    }

    @Transactional
    @Override
    public void editUser(long userID, String name, String surname, String email, Date birthDate, String address, String contact) {
        User u = findUserByID(userID);
        u.setName(name);
        u.setSurname(surname);
        u.setEmail(email);
        u.setBirthDate(birthDate);
        u.setAddress(address);
        u.setContact(contact);
        em.persist(u);
    }

    @Override
    public UserDetails findUserByUsername(String username) {
        List<User> useri  = em.createQuery("SELECT u FROM User u WHERE u.email = :username").setParameter("username", username).getResultList();
        return (User) em.createQuery("SELECT u FROM User u WHERE u.email = :username").setParameter("username", username).getResultList().get(0);
    }

    @Override
    public List<User> getUnapprovedUsers() {
        return em.createQuery("select u from User u where not u.enabled").getResultList();
    }

    @Override
    @Transactional
    public void approveUserProfile(User u) {
        u.setEnabled(true);
        em.persist(u);
        eventPublisher.publishEvent(new OnProfileEnabledEvent(u));
    }

    @Override
    public Role findById(Long id) {
        return em.find(Role.class, id);
    }

    @Override
    @Transactional
    public User updateUser(User s) {
        em.persist(s);
        return s;
    }

    @Override
    public User findByUsernameAndProvider(String username, String providers)
    {
        try
        {
            return (User) em.createQuery("select u from User u where u.email = :username and u.provider = :provider")
                    .setParameter("username", username)
                    .setParameter("provider", providers)
                    .getSingleResult();
        }
        catch (Exception ex)
        {
            return null;
        }
    }

    @Override
    @Transactional
    public User mergeUser(User user) {
        return em.merge(user);
    }

    @Override
    public List<User> getAdmins() {
        return em.createQuery("select u from User u where u.role.roleName = 'SUPERADMIN'")
                .getResultList();
    }

    @Override
    public List<User> findConnectedAccountsByUser(User u) {
        return em.createQuery("select u.connectedAccounts from User u where u = :user")
                .setParameter("user", u)
                .getResultList();
    }
}
