package com.tourMate.services.impl;


import com.tourMate.dao.TransportDao;
import com.tourMate.dto.*;
import com.tourMate.entities.*;
import com.tourMate.services.TransportManager;
import com.tourMate.services.UsersManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
public class TransportManagerImpl implements TransportManager {
    @Autowired
    TransportDao transportDao;

    @Autowired
    UsersManager usersManager;

    @Override
    public void createTransport(String transportName, String carBrand, String carType, int carManufacturedYear, int noPassengers, int noBags, long EMBG, Long userId, String carPlate) {
        transportDao.createTransport(transportName, carBrand, carType, carManufacturedYear, noPassengers, noBags, EMBG, userId, carPlate);
    }

    @Override
    public void deleteTransport(long transportId) {
        transportDao.deleteTransport(transportId);
    }

    @Override
    public List<Transport> getTransports() {
        return transportDao.getTransports();
    }

    @Override
    public List<TransportDto> getTransportsByUser(long userId) {
        return transportDao.getTransportsByUser(userId);
    }

    @Override
    public List<RouteListingDto> getRoutesForTransport(long transportId) {
        return transportDao.getRoutesForTransport(transportId);
    }

    @Override
    public TransportDto findTransportById(long transportId) {
        return transportDao.findTransportById(transportId);
    }

    @Override
    public Transport getTransportById(Long transportId) {
        return transportDao.getTransportById(transportId);
    }

    @Override
    public void editTransport(long transportID, String transportName, String carBrand, String carType, int carManufacturedYear, int noPassengers, int noBags, long EMBG, User owner, String carPlate) {
        transportDao.editTransport(transportID, transportName, carBrand, carType, carManufacturedYear, noPassengers, noBags, EMBG, owner, carPlate);
    }

    @Override
    public void createTransportReservation(Long transportRouteId, Long userId, int noSeats) {
        TransportRoute transportRoute = transportDao.findTransportRouteById(transportRouteId);
        User u = usersManager.findUserByID(userId);
        TransportReservation transportReservation = new TransportReservation(transportRoute, noSeats, u);
        transportDao.createTransportReservation(transportReservation);
    }

    @Override
    public TransportReservation findTransportReservationByID(long reservationID) {
        return transportDao.findTransportReservationByID(reservationID);
    }

    @Override
    public List<TransportReservation> getTransportReservations() {
        return transportDao.getTransportReservations();
    }

    @Override
    public TransportAvailible findTransportAvailableByID(long reservationID) {
        return transportDao.findTransportAvailableByID(reservationID);
    }

    @Override
    public Transport getTransportById(long transportId) {
        return transportDao.getTransportById(transportId);
    }

    @Override
    public List<Reviews> getReviewsForTransport(long transportId) {
        Transport transport = getTransportById(transportId);
        return transportDao.getReviewsForTransport(transport);
    }

    @Override
    public List<TransportListingDto> getTransportsAvailableByFilters(String from, String to, Date date, int numPassengers) {
        List<TransportRoute> transportAvailable = transportDao.getTransportsAvailableByFilters(from, to, date, numPassengers);
        Map<TransportAvailible, List<TransportRoute>> transportsByTransporter = transportAvailable.stream().collect(Collectors.groupingBy(x -> x.getParentRoute()));
        List<TransportListingDto> transportList = transportsByTransporter.keySet().stream().toList().stream()
                .map(x -> new TransportListingDto(
                        x.getTransportAvailibleId(),
                        x.getFrom(),
                        x.getTo(),
                        x.getDate(),
                        x.getFreeSpace(),
                        x.getTime(),
                        transportsByTransporter.get(x).stream().mapToDouble(y -> y.getPrice()).min().getAsDouble(),
                        x.getRoutes(),
                        x.getTransport(),
                        getReviewsForTransport(x.getTransport().getTransportID()),
                        getReviewsForTransport(x.getTransport().getTransportID()).stream().mapToDouble(Reviews::getNumStar).average().orElse(0)
                )).collect(Collectors.toList());
        return transportList;
    }

    @Override
    public List<TransportReservation> getTransportsReservationsByUserID(long userID) {
        return transportDao.getTransportsReservationsByUserID(userID);
    }

    @Override
    public List<TransportAvailible> getTransportsAvailable() {
        return transportDao.getTransportsAvailable();
    }

    @Override
    public void createTransportAvailable(Transport transport, String departureLocation, String arrivalLocation, Date date, Integer noSeats, Date departureHour, Collection<TransportRoute> routes) {
        transportDao.createTransportAvailable(transport, departureLocation, arrivalLocation, date, noSeats, departureHour, routes);
    }

    @Override
    public void editTransportReservation(Transport transport, long reservationID, String departureLocation, String arrivalLocation, Date date, Integer noSeats, User user, Date departureHour) {
        transportDao.editTransportReservation(transport, reservationID, departureLocation, arrivalLocation, date, noSeats, user, departureHour);
    }

    @Override
    public void editTransportAvailable(Transport transport, long availableID, String departureLocation, String arrivalLocation, Date date, Integer noSeats, Date departureHour) {
        transportDao.editTransportAvailable(transport, availableID, departureLocation, arrivalLocation, date, noSeats, departureHour);
    }

    @Override
    public void deleteTransportReservation(long reservationID) {
        transportDao.deleteTransportReservation(reservationID);
    }

    @Override
    public void deleteTransportAvailable(long availableID) {
        transportDao.deleteTransportAvailable(availableID);
    }

    @Override
    public void createTransportRoute(TransportAvailible parentRoute, String from, String to, double price, Date departure, Date arrival, int freeSpace, int order) {
        transportDao.createTransportRoute(parentRoute, from, to, price, departure, arrival, freeSpace, order);
    }

    @Override
    public void deleteTransportRoute(long transportRouteId) {
        transportDao.deleteTransportRoute(transportRouteId);
    }

    @Override
    public void editTransportRoute(long transportRouteId, TransportAvailible parentRoute, String from, String to, double price, Date departure, Date arrival, int freeSpace, int order) {
        transportDao.editTransportRoute(transportRouteId, parentRoute, from, to, price, departure, arrival, freeSpace, order);
    }

    @Override
    public List<TransportReservationUserDto> findValidTransportReservationsByUser(Long userId) {
        User u = usersManager.findUserByID(userId);
        List<TransportReservation> reservations = transportDao.findReservationByUser(u);
        return reservations.stream().map(x -> new TransportReservationUserDto(
                x.getUser(),
                x.getTransportRoute(),
                x.getTransportRoute().getParentRoute().getTransport().getTransportID(),
                x.getNoSeats(),
                x.getTransportRoute().getParentRoute().getTransport().getCarBrand(),
                x.getTransportRoute().getParentRoute().getTransport().getCarType(),
                x.getTransportRoute().getParentRoute().getTransport().getOwner(),
                x.getReviewed()
        )).toList();
    }

    @Override
    public List<TransportReservationUserDto> findPastTransportReservationsByUser(Long userId) {
        User u = usersManager.findUserByID(userId);
        List<TransportReservation> reservations = transportDao.findPastReservationByUser(u);
        return reservations.stream().map(x -> new TransportReservationUserDto(
                x.getUser(),
                x.getTransportRoute(),
                x.getTransportRoute().getParentRoute().getTransport().getTransportID(),
                x.getNoSeats(),
                x.getTransportRoute().getParentRoute().getTransport().getCarBrand(),
                x.getTransportRoute().getParentRoute().getTransport().getCarType(),
                x.getTransportRoute().getParentRoute().getTransport().getOwner(),
                x.getReviewed()
        )).toList();
    }

}


